//-----------------------------------------------------------------------------
// Class name: LocksLoggerThread
// Class description: the class implements a thread that keeps truck of every lock operation that is made in the system.		 
//		 The class is implemented as singelton, meaning, there could be only one object of this class in the application
//		 To get a pointer to this object call the method GetInstance()
// Note: The class is not multithreaded safe
//--------------------------------------------------------------------------------
#include "LocksLoggerThread.h"
#include <process.h>

// duplicated typedef - also defined at hookedFunctions.cpp. to remove later ???!!!
typedef DWORD  (WINAPI *WAITFORSINGLEOBJECTPROC)(HANDLE, DWORD);
// ptr to original functions - initialized at hookedFunctions.cpp file
extern WAITFORSINGLEOBJECTPROC	g_pfnOrigWaitForSingleObject;


LocksLoggerThread::LocksLoggerThread()  :
	m_hThread(NULL),
	m_dwThreadId(0),
	m_bRun(false)
{

}

LocksLoggerThread::~LocksLoggerThread()
{
	m_bRun = false;
	WaitForSingleObject(m_hThread, INFINITE);
	CloseHandle(m_hThread);
	m_hThread = NULL;
}

/*-----------------------------------------------------------------------
Name: run
Description: run the thread
Return:		 0 - success, else - failure
------------------------------------------------------------------------*/
HRESULT LocksLoggerThread::run()
{
	if (true == m_bRun)
		return S_OK;

	m_bRun = true;

	if (NULL == (m_hThread = CreateThread(NULL, 0, threadFunc, (LPVOID)this, 0, &m_dwThreadId)))
	{
		m_bRun = false;;
		return (errno != S_OK ? errno : S_FALSE);
	}
	
	// Create msg queue for the thread
	while (PostThreadMessage(m_dwThreadId, WM_LM_INIT_Q, (WPARAM)0, (LPARAM)0) == FALSE)
	{
		int error = GetLastError();
		Sleep(1000);
	}

	return S_OK;
}

/*-----------------------------------------------------------------------
Name: stop
Description: stop the thread
------------------------------------------------------------------------*/
void LocksLoggerThread::stop()
{
	m_bRun = false;
	PostThreadMessage(m_dwThreadId, WM_QUIT, (WPARAM)0, (LPARAM)0);
	WaitForSingleObject(m_hThread, 30000);
}

/*-----------------------------------------------------------------------
Name: threadFunc
Description: static method that runs the thread
Parameters:  param - pointer to this object
Return:		 0 - success, else - failure
------------------------------------------------------------------------*/
DWORD WINAPI LocksLoggerThread::threadFunc(LPVOID param)
{
	LocksLoggerThread* pObject = (LocksLoggerThread*)param;

	return pObject->threadMemberFunc();
}

/*-----------------------------------------------------------------------
Name: threadMemberFunc
Description: the thread method. This mehod actually runs in a seperate thread
Return:		 0 - success, else - failure
------------------------------------------------------------------------*/
DWORD LocksLoggerThread::threadMemberFunc()
{
	MSG msg;
	
	while (m_bRun && GetMessage(&msg, NULL, 0, WM_LM_QUIT)) 
	{
		switch (msg.message)
		{
			case (WM_LM_IGNORE):
				m_ignoredObjMap[((LockData*)(msg.wParam))->getHandle()] = ((LockData*)(msg.wParam))->getHandle();
				delete ((LockData*)(msg.wParam));
				break;

			case(WM_LM_LOCK):
			case(WM_LM_UNLOCK):
				// a lock operation was made (lock or unlock)
				addLockDataToLockList(*((LockData*)(msg.wParam)));
				delete ((LockData*)(msg.wParam));
				break;
			
			case(WM_LM_INIT_Q):
				// dummy message to init message queue for this thread
				break;

			case(WM_LM_QUIT):
				// exit the thread
				return 0;
				break;
			
			default:
				// no operation
				break;
		}
	}

	removeIgnoredObjects();
	return 0;
}

/*-----------------------------------------------------------------------
Name: addLockDataToLockList
Description: add the lock information to m_thread2LockListMap
Parameters:  lockData - the lock info to add 
------------------------------------------------------------------------*/
void LocksLoggerThread::addLockDataToLockList(const LockData& lockData)
{
	ThreadID2LockDataListMap::iterator lockDataMapIter = m_thread2LockListMap.find(lockData.getThreadID());

	if (lockDataMapIter == m_thread2LockListMap.end())
	{
		LockDataList lockDataList;
		m_thread2LockListMap[lockData.getThreadID()] = lockDataList;
		lockDataMapIter = m_thread2LockListMap.find(lockData.getThreadID());
	}

	LockDataList& lockDataList = ((*lockDataMapIter).second);
	lockDataList.push_back(lockData);
}

/*-----------------------------------------------------------------------
Name: removeIgnoredObjects
Description: remove from m_thread2LockListMap all objects that are in m_ignoredObjMap.
			 in this way we can remove unwanted objects, in our case: event objects
------------------------------------------------------------------------*/
void LocksLoggerThread::removeIgnoredObjects()
{
	HANDLE hIgnoredObject = NULL;
	HANDLE hObject = NULL;

	// loop over all ignored objects
	for (Obj2ObjMap::const_iterator ignoredObjIter = m_ignoredObjMap.begin();
		 ignoredObjIter != m_ignoredObjMap.end();
		 ignoredObjIter++)
	{
		hIgnoredObject = (*ignoredObjIter).second;
		// loop over all threads
		for (ThreadID2LockDataListMap::iterator mapIter = m_thread2LockListMap.begin();
			 mapIter != m_thread2LockListMap.end();
			 mapIter++)
		{
			// loop over list of locks for a specific thread
			LockDataList& lockDataList = ((*mapIter).second);
			LockDataList::iterator erasedElemIter;
			for (LockDataList::iterator listIter = lockDataList.begin();
				 listIter != lockDataList.end();
				 listIter++)
			{
				LockData& lockData = (*listIter);
				hObject = lockData.getHandle();
				if (hObject == hIgnoredObject)
				{
					erasedElemIter = listIter;
					listIter--;
					lockDataList.erase(erasedElemIter);
				}
			}
		}
	}
}

/*-----------------------------------------------------------------------
Name: WaitForSingleObject
Description: unhooked version of WaitForSingleObject
Return:		 return code from original WaitForSingleObject()
------------------------------------------------------------------------*/
DWORD LocksLoggerThread::WaitForSingleObject(HANDLE hObject, DWORD dwMilliseconds)
{
	if (NULL != g_pfnOrigWaitForSingleObject)
		return g_pfnOrigWaitForSingleObject(hObject, dwMilliseconds);
	
	return ::WaitForSingleObject(hObject, dwMilliseconds);
}