//-----------------------------------------------------------------------------
// Class name: DDAMgr
// Class description: the class implements the manager of locks mentor.
//		 The class responsible for the flow of the locks mentor.
//		 The class is implemented as singelton, meaning, there could be only one object of this class in the application
//		 To get a pointer to this object call the method GetInstance()
// Note: The class is not multithreaded safe
//--------------------------------------------------------------------------------
#include <iostream.h>
#include "DDAMgr.h"
#include "LocksLoggerThread.h"
#include "hookedFunctions.h"
#include "LockSequenceBuilder.h"
#include "LockSequenceAnalyzer.h"

/*-----------------------------------------------------------------------
						Constructor
------------------------------------------------------------------------*/
DDAMgr::DDAMgr() :
	m_nState(eStart)
{

}

/*-----------------------------------------------------------------------
						Destructor
------------------------------------------------------------------------*/
DDAMgr::~DDAMgr()
{

}

/*-----------------------------------------------------------------------
Name: start
Description: start to run the locks logger (the locks logger log all lock operations
			 done in the application) and hook all synchronization functions
Parameters:
Return: 0 - success, else - failure
------------------------------------------------------------------------*/
HRESULT DDAMgr::start()
{
	LocksLoggerThread* pLogger = LocksLoggerThread::getInstance();

	if (eStart != m_nState)
	{
		return ERROR_INVALID_STATE;
	}

	if (FALSE == hook())
	{
		m_nState = eError;
		return S_FALSE;
	}
	
	if (S_OK != pLogger->run())
	{
		m_nState = eError;
		return S_FALSE;
	}

	m_nState = eStop;
	return S_OK;
}

/*-----------------------------------------------------------------------
Name: stop
Description: stop the locks logger (the locks logger log all lock operations
			 done in the application) and unhook hooked functions
Return: 0 - success, else - failure
------------------------------------------------------------------------*/
HRESULT DDAMgr::stop()
{
	if (eStop != m_nState)
	{
		return ERROR_INVALID_STATE;
	}

	unhook();

	LocksLoggerThread* pLogger = LocksLoggerThread::getInstance();
	pLogger->stop();

	m_nState = eAnalyze;
	
	return S_OK;
}

/*-----------------------------------------------------------------------
Name: analyze
Description: take the info collected by the locks logger and analyze it (search
			 for possible deadlocks)
Return: 0 - success, else - failure
------------------------------------------------------------------------*/
HRESULT DDAMgr::analyze()
{
	LocksLoggerThread* pLogger = LocksLoggerThread::getInstance();
	
	if (eAnalyze != m_nState)
	{
		return ERROR_INVALID_STATE;
	}


	LockSequenceList     lockSeqList;
	LockSequenceBuilder  builder;
	LockSequenceAnalyzer analyzer;

	builder.Build(pLogger->getThread2LockListMap(), lockSeqList);

	/*cout << endl;

	for (LockSequenceList::const_iterator lockSeqListIter = lockSeqList.begin();
		 lockSeqListIter != lockSeqList.end();
		 lockSeqListIter++)
	{
		for (LockSequence::const_iterator lockSeqIter = lockSeqListIter->begin();
			 lockSeqIter != (*lockSeqListIter).end();
			 lockSeqIter++)
		{
			cout << lockSeqIter->getHandle() << " ";
		}

		cout << endl;
	}*/

	
	analyzer.analyze(lockSeqList, m_deadlockSeqList); 
	m_nState = eLastState;
	return S_OK;
}

/*-----------------------------------------------------------------------
Name: terminate
Description: terminate the DDA. On termination search for potential deadlocks
Return: 0 - success, else - failure
------------------------------------------------------------------------*/
HRESULT DDAMgr::terminate()
{
	switch (m_nState)
	{
	case (DDAMgr::eStop):
		stop();
		// no break - intent
	case (DDAMgr::eAnalyze):
		analyze();
		break;
	default:
		return S_FALSE;
		break;
	}

	return S_OK;
}