//-----------------------------------------------------------------------------
// Class name: DDAViewThread
// Class description: this class handles all the GUI issues.
//		  It acts as the presentation layer for the DDAMgr class
//--------------------------------------------------------------------------------
#include "DDAViewThread.h"
#include "resource.h"
#include "symServer.h"


extern SymServer* g_pSymServer;

/*-----------------------------------------------------------------------
						Constructor
------------------------------------------------------------------------*/
DDAViewThread::DDAViewThread()
{
	m_pMgr = DDAMgr::getInstance();
}

/*-----------------------------------------------------------------------
						Destructor
------------------------------------------------------------------------*/
DDAViewThread::~DDAViewThread()
{

}

/*-----------------------------------------------------------------------
Name: showDlg
Description: create the modal dialog box
Parameters: none
Return: 0 - success, else - failure
------------------------------------------------------------------------*/
HRESULT DDAViewThread::showDlg() const
{
	// display main dialog 
	DialogBoxParam(GetModuleHandle("DDA.dll"), MAKEINTRESOURCE(IDD_DDA_DLG), NULL, DialogProc, (LPARAM)this);
	return S_OK;
}

/*-----------------------------------------------------------------------
Name: showResults
Description: show the results after finished analyzing the locks
Parameters: none
Return: 0 - success, else - failure
------------------------------------------------------------------------*/
void DDAViewThread::showResults() const
{
	if (DDAMgr::eLastState != m_pMgr->getState())
	{
		MessageBox(NULL, "Results are available only after analyzing the locks", "DDA Results", MB_OK); 
		return;
	}

	const LockSequenceList& lockSeqList = m_pMgr->getDeadlockSeqList();

	if (0 == lockSeqList.size())
	{
		MessageBox(NULL, "no potential deadlock", "DDA Results", MB_OK); 
		return;
	}

	// display deadlock flow
	MessageBox(NULL, "potential deadlock was detected\npress o.k. button for more details", "DDA Results", MB_OK);

	char  szResults[8192] = "";
	char  szSymbols[1024] = "";
	DWORD dwCallerAddr = 0;

	for (LockSequenceList::const_iterator lockSeqListIter = lockSeqList.begin();
		 lockSeqListIter != lockSeqList.end();
		 lockSeqListIter++)
	{
		for (LockSequence::const_iterator lockSeqIter = lockSeqListIter->begin();
			 lockSeqIter != lockSeqListIter->end();
			 lockSeqIter++)
		{
			szSymbols[0] = 0;			

			dwCallerAddr = lockSeqIter->getCallerAddr();
			g_pSymServer->AddrToSym(dwCallerAddr, szSymbols, sizeof(szSymbols));
			_snprintf(szResults+strlen(szResults), sizeof(szResults)-strlen(szResults), "handle: 0x%x, thread: 0x%x, position: %s\n",
					  lockSeqIter->getHandle(),
					  lockSeqIter->getThreadID(),
					  szSymbols);
			szResults[sizeof(szResults)-1] = 0;
		}
		
		_snprintf(szResults+strlen(szResults), sizeof(szResults)-strlen(szResults), "------------------\n");
	}

	MessageBox(NULL, szResults, "DDA Results", MB_OK);
}

/*-----------------------------------------------------------------------
Name: terminate
Description: call this method before application exits
Parameters: none
Return: 0 - success, else - failure
------------------------------------------------------------------------*/
void DDAViewThread::terminate()
{
	if (S_OK == m_pMgr->terminate())
	{
		showResults();
	}
}

/*-----------------------------------------------------------------------
Name: doAction
Description: call this method whenever the user pushes the push button
Parameters: none
Return: 0 - success, else - failure
------------------------------------------------------------------------*/
HRESULT	DDAViewThread::doAction(HWND hDlg)
{
	HWND hWnd = GetDlgItem(hDlg, ID_ACTION_PB);
	
	EnableWindow(hWnd, FALSE);

	switch (m_pMgr->getState())
	{
	case (DDAMgr::eStart):
		if (S_OK == m_pMgr->start())
		{
			SetDlgItemText(hDlg, ID_ACTION_PB, "Stop");
		}
		else
		{
			SetDlgItemText(hDlg, ID_ACTION_PB, "Error-disabled!");
		}

		break;

	case (DDAMgr::eStop):
		m_pMgr->stop();
		SetDlgItemText(hDlg, ID_ACTION_PB, "Analyzing...");
		// no break - intent

	case (DDAMgr::eAnalyze):
		m_pMgr->analyze();
		showResults();
		SetDlgItemText(hDlg, ID_ACTION_PB, "Done");
		break;
	}

	EnableWindow(hWnd, TRUE);
	return S_OK;
}

/*-----------------------------------------------------------------------
Name: DialogProc
Description: processes messages sent to the modeless dialog box
Parameters: hDlg - handle to dialog box, uMsg - message, 
			wParam - first message parameter, lParam - second message parameter
Return: return TRUE if it processed the message, and FALSE if it did not
------------------------------------------------------------------------*/
BOOL CALLBACK DDAViewThread::DialogProc(HWND hDlg,		
										UINT uMsg,		
										WPARAM wParam,	
										LPARAM lParam)
{
	static DDAViewThread* pThis = NULL;
	DDAMgr* pMgr = DDAMgr::getInstance();

	switch (uMsg) 
	{
	case WM_INITDIALOG:
		pThis = (DDAViewThread*)lParam;
		SetDlgItemText(hDlg, ID_ACTION_PB, "Start");
		pThis->doAction(hDlg);
		return TRUE;
		break;		

	case WM_COMMAND:
		pThis->doAction(hDlg);
		break;

	case WM_CLOSE:
		if (IDYES == MessageBox(NULL, "Are you sure you want to exit?", "DDA", MB_YESNO | MB_ICONQUESTION | MB_DEFBUTTON2 | MB_TOPMOST))
		{
			EndDialog(hDlg, 0);
		}
		break;		
	}

	return FALSE;
}

/*-----------------------------------------------------------------------
Name: threadMemberFunc
Description: the thread method. This mehod actually runs in a seperate thread
Return:		 0 - success, else - failure
------------------------------------------------------------------------*/
DWORD DDAViewThread::threadMemberFunc()
{
	showDlg();

	return 0;
}