//-----------------------------------------------------------------------------
// Class name: SymServer
// Class description: handles translation of addresses to meaningfull symboles.
//		  It is being used when translating the addresses in the stack trace.
//--------------------------------------------------------------------------------
#pragma warning(disable:4786)
#include "symServer.h"
#include <iostream.h>
#include <stdio.h>
#include <vector>
#include <string>

#define MAXNAMELEN 1024
#define MAX_SEARCH_PATH 8192

HANDLE SymServer::m_hProcess = GetCurrentProcess();
bool   SymServer::m_bInit = false;

SymServer* SymServer::Create() 
{ 
	SymServer* pSymServer = new SymServer(); 
	if (pSymServer)
	{
		if (S_FALSE == pSymServer->Init())
		{
			delete pSymServer;
			pSymServer = NULL;
		}
	}

	return pSymServer;
}

SymServer::SymServer()
{
	// init line struct
	memset( &m_line, '\0', sizeof(m_line));
	m_line.SizeOfStruct = sizeof(m_line);
	
	// init symbole struct
    if ((m_pSymbol = (PIMAGEHLP_SYMBOL)(new BYTE[sizeof(IMAGEHLP_SYMBOL) + 512])) == NULL)
		return;
	memset(m_pSymbol, 0, sizeof(IMAGEHLP_SYMBOL) + 512);
    m_pSymbol->SizeOfStruct = sizeof(IMAGEHLP_SYMBOL);
    m_pSymbol->MaxNameLength = 512;
	
	// init module struct
	memset(&m_module, '\0', sizeof(m_module));
	m_module.SizeOfStruct = sizeof(m_module);
}

SymServer::~SymServer()
{
	SymCleanup(m_hProcess);
	delete []m_pSymbol;
}

HRESULT SymServer::Init()
{
	if (true == m_bInit)
		return S_OK;

	// set sym options
	DWORD dwSymOpt = 0;

	dwSymOpt  = SymGetOptions();
	dwSymOpt |= SYMOPT_DEFERRED_LOADS; // enahnced perfomance - load symboles uppon demand
	dwSymOpt |= SYMOPT_LOAD_LINES;     // load line info
	dwSymOpt &= ~SYMOPT_UNDNAME;       // show undecorated name
	SymSetOptions(dwSymOpt);

	if (FALSE == SymInitialize(m_hProcess, NULL, TRUE))
        return S_FALSE;
	
	// add the executable directory to the symboles search path
	TCHAR szExeDirPath[_MAX_PATH] = "";
	if (!GetModuleFileName(NULL, szExeDirPath, sizeof(szExeDirPath)))
		return S_FALSE;
	
	char *pszSlash = _tcsrchr(szExeDirPath, '\\');
	if (pszSlash)
		*pszSlash = 0;

	TCHAR szCurrSearchPath[MAX_SEARCH_PATH] = "";
	if (FALSE == SymGetSearchPath(m_hProcess, szCurrSearchPath, sizeof(szCurrSearchPath)))
		return S_FALSE;

	std::string strNewSearchPath(szExeDirPath + std::string(";") + szCurrSearchPath);
	SymSetSearchPath(m_hProcess, (LPTSTR)strNewSearchPath.c_str()); 

	m_bInit = true;
	return S_OK;
}

void SymServer::GetStack(PDWORD pStackArray, const int nArraySize, int& nStackDepth)
{
	CONTEXT		context;
	STACKFRAME	stackFrame;
	HANDLE		hThread = GetCurrentThread();
	
	nStackDepth = 0;

	memset(&context, 0, sizeof (context));
	context.ContextFlags = CONTEXT_FULL;
	GetThreadContext(hThread, &context);

	memset(&stackFrame, 0, sizeof(stackFrame));
	stackFrame.AddrPC.Offset    = context.Eip;
	stackFrame.AddrPC.Mode		= AddrModeFlat;
    stackFrame.AddrStack.Offset = context.Esp;
    stackFrame.AddrStack.Mode	= AddrModeFlat;
	stackFrame.AddrFrame.Offset = context.Ebp;
	stackFrame.AddrFrame.Mode	= AddrModeFlat;

	while (StackWalk(IMAGE_FILE_MACHINE_I386,
                     m_hProcess,
                     hThread,
                     &stackFrame,
                     &context,
                     0,
                     SymFunctionTableAccess,
                     SymGetModuleBase,
                     0))
    {
		
        if (stackFrame.AddrFrame.Offset == 0 ||
			nStackDepth >= nArraySize) 
		{
			break;
		}
		
		pStackArray[nStackDepth] = stackFrame.AddrPC.Offset;
		nStackDepth++;
    }	
}

void SymServer::AddrToSym(const DWORD dwAddr, LPTSTR lpszSymbols, const int nSymLen)
{
	TCHAR szFuncName[MAXNAMELEN] = "";
	TCHAR szFileName[MAXNAMELEN] = "";
	TCHAR szModuleName[MAXNAMELEN] = "";
	DWORD dwLine = 0;

	AddrToLine(dwAddr, dwLine);
	AddrToFile(dwAddr, szFileName, sizeof(szFileName));
	AddrToModule(dwAddr, szModuleName, sizeof(szModuleName));
	AddrToFunc(dwAddr, szFuncName, sizeof(szFuncName));

	_snprintf(lpszSymbols, nSymLen, "module = %s * func = %s * file = %s * line = %d", szModuleName, szFuncName, szFileName, dwLine);
}

void SymServer::AddrToLine(const DWORD dwAddr, DWORD& dwLine)
{
	DWORD dwDisplacement = 0;
	dwLine = 0;

	SymGetLineFromAddr(m_hProcess, dwAddr, &dwDisplacement, &m_line);
	dwLine = m_line.LineNumber;
}

void SymServer::AddrToFile(const DWORD dwAddr, LPTSTR lpszFile, const int nFileLen)
{
	DWORD dwDisplacement = 0;
	lpszFile[0] = 0;

	SymGetLineFromAddr(m_hProcess, dwAddr, &dwDisplacement, &m_line);
	if (NULL == m_line.FileName)
	{
		return;
	}

	strncpy(lpszFile, m_line.FileName, nFileLen);
	lpszFile[nFileLen-1] = 0;
}

void SymServer::AddrToFunc(const DWORD dwAddr, LPTSTR lpszFunc, const int nFuncLen)
{
	DWORD dwDisplacement = 0;
	lpszFunc[0] = 0;

	if (SymGetSymFromAddr(m_hProcess, dwAddr, &dwDisplacement, m_pSymbol))
    {
		if (!UnDecorateSymbolName(m_pSymbol->Name, lpszFunc, nFuncLen, UNDNAME_COMPLETE))
		{
			strncpy(lpszFunc, m_pSymbol->Name, nFuncLen);
			lpszFunc[nFuncLen-1] = 0;
		}
	}
}

void SymServer::AddrToModule(const DWORD dwAddr, LPTSTR lpszModule, const int nModuleLen)
{
	lpszModule[0] = 0;

	if (SymGetModuleInfo(m_hProcess, dwAddr, &m_module))
	{
		strncpy(lpszModule, m_module.ImageName, nModuleLen);
		lpszModule[nModuleLen-1] = 0;
	}
}

