import React from 'react';
import PropTypes from 'prop-types';
import {View} from 'react-native';
import MapView, {Marker, Callout} from 'react-native-maps';
import RouteMapSlotCard from "./RouteMapSlotCard";
import RouteMapCompanyCard from "./RouteMapCompanyCard";

class RouteMapSection extends React.Component {
    mapRef = null;
    state = {
        markers: []
    }

    componentDidMount() {
        if (this.props.slots) {
            const markers = this.props.slots.filter(s => this.hasAddress(s)).map(slot => this.getMarkerFromSlot(slot));

            this.setState({markers: markers});
            setTimeout(() => this.mapRef.fitToElements(true, {
                edgePadding: {
                    top: 10,
                    right: 10,
                    bottom: 10,
                    left: 10
                }
            }), 500);
        } else if (this.props.companies) {
            const markers = this.props.companies.filter(c => c.address !== undefined && c.address !== null).map(c => this.getMarkerFromCompany(c));

            this.setState({markers: markers});
            setTimeout(() => this.mapRef.fitToElements(true, {
                edgePadding: {
                    top: 10,
                    right: 10,
                    bottom: 10,
                    left: 10
                }
            }), 500);
        }
    }

    hasAddress(slot) {
        if (slot.company) {
            return slot.company.address && slot.company.address.position;
        }
        if (slot.school) {
            return slot.school.address && slot.school.address.position;
        }
        if (slot.location) {
            return slot.location.address && slot.location.address.position;
        }
        return false;
    }

    getDestination(slot) {
        if (slot.company) {
            return slot.company;
        }
        if (slot.school) {
            return slot.school;
        }
        return slot.location;
    }

    getMarkerFromSlot(slot) {
        const destination = this.getDestination(slot);
        const coordinates = destination.address.position.coordinates;

        return <Marker
            key={slot.id}
            coordinate={{latitude: coordinates[1], longitude: coordinates[0]}}
        >
            <Callout
                onPress={() => {
                    if (slot.company) {
                        this.props.onSelectCompany(destination.id);
                    }
                }}>
                <RouteMapSlotCard slot={slot} destination={destination}
                                  isCompany={slot.company !== null}/>
            </Callout>
        </Marker>
    }

    getMarkerFromCompany(company) {
        const coordinates = company.address.position.coordinates;

        return <Marker
            key={company.id}
            tracksViewChanges={false}
            coordinate={{latitude: coordinates[1], longitude: coordinates[0]}}
        >
            <Callout
                onPress={() => this.props.onSelectCompany(company.id)}>
                <RouteMapCompanyCard company={company} />
            </Callout>
        </Marker>
    }

    render() {
        return (
            <View style={{flex: 1, alignItems: "stretch"}}>
                <MapView
                    ref={(ref) => {
                        this.mapRef = ref
                    }}
                    initialRegion={{
                        latitude: 51.765,
                        longitude: 5.51806,
                        latitudeDelta: 0.04,
                        longitudeDelta: 0.04,
                    }} style={{height: 400}}
                >
                    {this.state.markers}
                </MapView>
            </View>
        );
    }
}

RouteMapSection.propTypes = {
    navigation: PropTypes.object,
    slots: PropTypes.array,
    companies: PropTypes.array,
    onSelectCompany: PropTypes.func,
};

export default RouteMapSection;
