import React from 'react';
import PropTypes from 'prop-types';
import {
  ScrollView,
  Text,
  View,
  Platform,
  UIManager,
  RefreshControl,
  FlatList,
} from 'react-native';
import {Card} from 'react-native-elements';
import {connect} from 'react-redux';
import Screen from '../../components/Screen';
import {
  companiesGetById,
  companiesSetShown,
} from '../../shared/actions/companies';
import {routesGetForGroup} from '../../shared/actions/routes';
import {pollsGet, pollsSend, pollsGetById} from '../../shared/actions/polls';
import {bindActionCreators} from 'redux';
import RaisedButton from '../../components/RaisedButton';
import Moment from 'moment/min/moment-with-locales';
import nl from 'javascript-time-ago/locale/nl/index';
import {showMessage} from 'react-native-flash-message/src/index';
import Icon from 'react-native-vector-icons/EvilIcons';
import NavigationService from '../../shared/services/NavigationService';

class FeedbackScreen extends Screen {
  constructor() {
    super();
    Moment.locale('nl');

    if (Platform.OS === 'android') {
      UIManager.setLayoutAnimationEnabledExperimental &&
        UIManager.setLayoutAnimationEnabledExperimental(true);
    }
  }

  componentDidMount() {
    this.props.pollsGet(this.props.auth.user.id);
  }

  goToFeedbackDetail(pollId) {
    this.props.pollsGetById(this.props.auth.user.id, pollId);
    NavigationService.navigate('feedbackDetail', {isDetail: true});
  }

  componentDidUpdate(oldState) {
    if (
      this.props.polls.error &&
      oldState.polls.error !== this.props.polls.error
    ) {
      showMessage({
        message: this.props.polls.error,
        type: 'danger',
      });
    }
  }

  renderPollItem(poll) {
    return (
      <Card
        key={poll.id}
        title={poll.poll.title}
        style={{padding: 0, alignItems: 'center'}}>
        <View>
          <Text>{poll.poll.description || ''}</Text>
          <RaisedButton
            onPress={() => this.goToFeedbackDetail(poll.id)}
            title={'Invullen'}
          />
        </View>
      </Card>
    );
  }

  render() {
    const components = (
      <>
        {this.props.polls.list.length > 0 ? (
          <View style={{marginBottom: 30}}>
            <Card style={{padding: 0, alignItems: 'center'}}>
              <Text>
                Na het bezoeken van een bedrijf en na het evenement willen we
                graag weten wat jij er van vindt. We stellen je dan een paar
                korte vragen, invullen duurt nog geen minuut!
              </Text>
            </Card>
            {this.props.polls.list.map((poll) => this.renderPollItem(poll))}
          </View>
        ) : (
          <View style={{marginBottom: 30}}>
            <Card
              style={{
                padding: 0,
                flexDirection: 'column',
                justifyContent: 'start',
                alignItems: 'stretch',
              }}>
              <Icon
                name={'check'}
                size={100}
                color={'#999'}
                style={{textAlign: 'center', margin: 20}}
              />
              <Text style={{textAlign: 'center'}}>
                We hebben momenteel geen vragen voor je!
              </Text>
            </Card>
          </View>
        )}
      </>
    );

    return (
      <FlatList
        style={{flex: 1}}
        refreshControl={
          <RefreshControl
            refreshing={this.props.loading}
            onRefresh={() => this.props.pollsGet(this.props.auth.user.id)}
          />
        }
        data={[]}
        renderItem={() => null}
        ListFooterComponent={components}
      />
    );
  }
}

FeedbackScreen.propTypes = {
  dispatch: PropTypes.func,
  navigation: PropTypes.object,
  auth: PropTypes.object,
};

const mapStateToProps = (state) => {
  return {
    auth: state.auth,
    permissions: state.auth.permissions,
    companies: state.companies.list,
    categories: state.categories.list,
    shownCompany: state.companies.shownCompany,
    route: state.routes.route,
    event: state.auth.permissions.events[0],
    polls: state.polls,
    loading: state.polls.loading,
  };
};

const mapDispatchToProps = (dispatch) => {
  return bindActionCreators(
    {
      companiesGetById,
      pollsSend,
      pollsGet,
      pollsGetById,
      companiesSetShown,
      routesGetForGroup,
    },
    dispatch,
  );
};

export default connect(mapStateToProps, mapDispatchToProps)(FeedbackScreen);
