import React from 'react';
import PropTypes from 'prop-types';
import {
  ScrollView,
  FlatList,
  RefreshControl,
  View,
  Text,
  TouchableOpacity,
} from 'react-native';
import {Card} from 'react-native-elements';
import {connect} from 'react-redux';
import Screen from '../../components/Screen';
import {groupsGetAllMembers} from '../../shared/actions/groups';
import Icon from 'react-native-vector-icons/FontAwesome';
import Communications from 'react-native-communications';

import {bindActionCreators} from 'redux';

class GroupScreen extends Screen {
  componentDidMount() {
    const group = this.props.group;
    this.props.groupsGetAllMembers(
      group.school.organisation.id,
      group.school.id,
      group.id,
    );
  }

  renderRow({item}) {
    const hasPerson = item.person !== null && item.person !== undefined;
    const cardTitle =
      hasPerson &&
      [item.person.firstName, item.person.insertion, item.person.lastName].join(
        '',
      ).length > 0
        ? [
            item.person.firstName,
            item.person.insertion,
            item.person.lastName,
          ].join(' ')
        : item.username;

    const rowStyle = {
      flexDirection: 'row',
      alignItems: 'center',
    };
    const iconWrapperStyle = {
      width: 25,
    };

    return (
      <Card
        title={
          cardTitle +
          (item.accessLevel.includes('Guide') ? ' (Begeleider)' : '')
        }>
        {hasPerson && item.person.phoneNumber ? (
          <View style={rowStyle}>
            <View style={iconWrapperStyle}>
              <Icon name={'phone'} />
            </View>
            <TouchableOpacity
              onPress={() =>
                Communications.phonecall(item.person.phoneNumber, true)
              }>
              <Text style={{color: '#008cd6'}}>{item.person.phoneNumber}</Text>
            </TouchableOpacity>
          </View>
        ) : null}
        {hasPerson &&
        [
          item.person.firstName,
          item.person.insertion,
          item.person.lastName,
        ].join('').length === 0 ? (
          <View style={rowStyle}>
            <View style={iconWrapperStyle}>
              <Icon name={'user'} />
            </View>
            <Text>{item.username}</Text>
          </View>
        ) : null}
      </Card>
    );
  }

  onRefresh() {
    const group = this.props.group;
    this.props.groupsGetAllMembers(
      group.school.organisation.id,
      group.school.id,
      group.id,
    );
  }

  render() {
    return (
      <FlatList
        style={{flex: 1}}
        refreshControl={
          <RefreshControl
            refreshing={this.props.loading}
            onRefresh={() => this.onRefresh()}
          />
        }
        data={[]}
        renderItem={() => null}
        ListFooterComponent={
          <>
            <View style={{marginBottom: 20}}>
              <Card>
                <Text>
                  Je bent lid van groep {this.props.group.name} en hebt daarin
                  de rol {this.props.group.accessLevelPretty}. Er zitten nog{' '}
                  {this.props.members.length - 1} anderen in deze groep.
                </Text>
              </Card>
              <FlatList
                data={this.props.members}
                renderItem={(item) => this.renderRow(item)}
                keyExtractor={(item) => item.id}
              />
            </View>
          </>
        }
      />
    );
  }
}

GroupScreen.propTypes = {
  dispatch: PropTypes.func,
  navigation: PropTypes.object,
  auth: PropTypes.object,
  members: PropTypes.array,
};

const mapStateToProps = (state) => {
  return {
    auth: state.auth,
    loading: state.groups.loading,
    members: state.groups.members,
    group: state.auth.permissions.groups[0],
  };
};

const mapDispatchToProps = (dispatch) => {
  return bindActionCreators({groupsGetAllMembers}, dispatch);
};

export default connect(mapStateToProps, mapDispatchToProps)(GroupScreen);
