import React from 'react';
import PropTypes from 'prop-types';
import {ScrollView, FlatList, RefreshControl, Text} from 'react-native';
import {List, ListItem} from 'react-native-elements';
import {connect} from 'react-redux';
import Screen from '../../components/Screen';
import * as moment from 'moment/min/moment-with-locales.min';

import TimeAgo from 'javascript-time-ago';
import nl from 'javascript-time-ago/locale/nl';
import {messagesGetAll, messagesGetById} from '../../shared/actions/messages';
import {bindActionCreators} from 'redux';
import NavigationService from '../../shared/services/NavigationService';
import Avatar from '../../components/Avatar';

TimeAgo.addLocale(nl);
moment.locale('nl');
const timeAgo = new TimeAgo('nl');

class MessagesScreen extends Screen {
  componentDidMount() {
    if (this.props.auth.user) {
      this.props.messagesGetAll(this.props.auth.user.id);
    }
    TimeAgo.addLocale(nl);
    moment.locale('nl');
    this.timeAgo = new TimeAgo('nl');
  }

  componentDidUpdate(oldState) {
    if (this.props.auth.user && !oldState.auth.user) {
      this.props.messagesGetAll(this.props.auth.user.id);
    }
  }

  renderRow({item}) {
    console.log(this.timeAgo);
    return (
      <ListItem
        titleStyle={{fontWeight: item.read ? 'normal' : 'bold'}}
        subtitleStyle={{fontWeight: 'normal'}}
        onPress={() => this.selectMessage(item)}
        avatar={<Avatar message={item} />}
        title={item.subject}
        subtitle={
          timeAgo.format(new Date(item.receivedAt)) + ' - ' + item.sender.name
        }
      />
    );
  }

  selectMessage(message) {
    this.props.messagesGetById(this.props.auth.user.id, message.id);
    NavigationService.navigate('messagesDetail', {
      messageId: message.id,
      isDetail: true,
    });
  }

  onRefresh() {
    this.props.messagesGetAll(this.props.auth.user.id);
  }

  render() {
    const components = (
      <>
        {this.props.messages.length > 0 ? (
          <List containerStyle={{marginTop: 0, borderTopWidth: 0}}>
            <FlatList
              data={this.props.messages}
              renderItem={(item) => this.renderRow(item)}
              keyExtractor={(item) => item.id}
            />
          </List>
        ) : (
          <Text style={{padding: 20}}>
            Je hebt nog geen berichten ontvangen.
          </Text>
        )}
      </>
    );

    return (
      <FlatList
        refreshControl={
          <RefreshControl
            refreshing={this.props.loading}
            onRefresh={() => this.onRefresh()}
          />
        }
        data={[]}
        renderItem={() => null}
        ListFooterComponent={components}
      />
    );
  }
}

MessagesScreen.propTypes = {
  dispatch: PropTypes.func,
  navigation: PropTypes.object,
  auth: PropTypes.object,
  messages: PropTypes.array,
};

const mapStateToProps = (state) => {
  return {
    auth: state.auth,
    messages: state.messages.list,
    loading: state.messages.loading,
  };
};

const mapDispatchToProps = (dispatch) => {
  return bindActionCreators({messagesGetAll, messagesGetById}, dispatch);
};

export default connect(mapStateToProps, mapDispatchToProps)(MessagesScreen);
