﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using Microsoft.SqlServer.Management.Smo;
using System.Data;
using System.IO;
using Microsoft.SqlServer.Management.Nmo;
using System.Reflection;

namespace SQLScripter
{
    class ScripterHelper
    {
        private string _serverName;
        private string _databaseName;
        private string _path;
        private Server _server;
        private Database _db;
        private DateTime _lastUpdate = DateTime.MinValue;
        private DateTime _newLastUpdate = DateTime.MinValue;
        private string _lastUpdateFilename;

        public ScripterHelper(string serverName, string databaseName, string path)
        {
            _serverName = serverName;
            _databaseName = databaseName;
            _path = path;

            Console.WriteLine("Connecting database (Server Name = '{0}', Database Name = '{1}')...", serverName, databaseName);

            _server = new Server(serverName);
            _db = _server.Databases[databaseName];

            _lastUpdateFilename = Path.Combine(Path.GetDirectoryName(Assembly.GetExecutingAssembly().Location), serverName + "-" + databaseName + ".txt");

            if (File.Exists(_lastUpdateFilename))
            {
                _lastUpdate = DateTime.FromBinary(Convert.ToInt64(File.ReadAllText(_lastUpdateFilename)));
                _newLastUpdate = _lastUpdate;
            }
        }

        public void ScriptTables()
        {
            Console.Write("Scripting Tables...");

            Scripter scripter = new Scripter();
            scripter.Server = _server;
            scripter.Options.IncludeHeaders = false;
            scripter.Options.SchemaQualify = true;
            scripter.Options.SchemaQualifyForeignKeysReferences = true;
            scripter.Options.NoCollation = true;
            scripter.Options.DriAllConstraints = true;
            scripter.Options.DriAll = true;
            scripter.Options.DriAllKeys = true;
            scripter.Options.DriIndexes = true;
            scripter.Options.ClusteredIndexes = true;
            scripter.Options.NonClusteredIndexes = true;
            scripter.Options.ToFileOnly = true;

            string tablesPath = Path.Combine(_path, "Tables");

            Dictionary<string, Table> sqlTables = new Dictionary<string, Table>();

            foreach (Table table in _db.Tables)
            {
                if (!table.IsSystemObject)
                {
                    sqlTables.Add(table.Name, table);
                }
            }

            if (!Directory.Exists(tablesPath))
            {
                Directory.CreateDirectory(tablesPath);
            }
            else
            {
                string[] files = Directory.GetFiles(tablesPath, "*.sql");

                foreach (string file in files)
                {
                    if (!sqlTables.ContainsKey(Path.GetFileNameWithoutExtension(file)))
                        File.Delete(file);
                }
            }

            int index = 0;

            foreach (KeyValuePair<string, Table> kvp in sqlTables)
            {
                Table table = kvp.Value;

                if (table.DateLastModified > _lastUpdate)
                {
                    if (table.DateLastModified > _newLastUpdate)
                        _newLastUpdate = table.DateLastModified;

                    scripter.Options.FileName = Path.Combine(tablesPath, table.Name + ".sql");
                    scripter.Script(new Urn[] { table.Urn });
                    index++;
                }
            }

            Console.WriteLine(index);
        }

        public void ScriptViews()
        {
            Console.Write("Scripting Views...");

            Scripter scripter = new Scripter();
            scripter.Server = _server;
            scripter.Options.IncludeHeaders = false;
            scripter.Options.SchemaQualify = true;
            scripter.Options.SchemaQualifyForeignKeysReferences = true;
            scripter.Options.NoCollation = true;
            scripter.Options.DriAllConstraints = true;
            scripter.Options.DriAll = true;
            scripter.Options.DriAllKeys = true;
            scripter.Options.DriIndexes = true;
            scripter.Options.ClusteredIndexes = true;
            scripter.Options.NonClusteredIndexes = true;
            scripter.Options.ToFileOnly = true;

            string viewsPath = Path.Combine(_path, "Views");

            Dictionary<string, View> sqlViews = new Dictionary<string, View>();

            foreach (View view in _db.Views)
            {
                if (!view.IsSystemObject)
                {
                    sqlViews.Add(view.Name, view);
                }
            }

            if (!Directory.Exists(viewsPath))
            {
                Directory.CreateDirectory(viewsPath);
            }
            else
            {
                string[] files = Directory.GetFiles(viewsPath, "*.sql");

                foreach (string file in files)
                {
                    if (!sqlViews.ContainsKey(Path.GetFileNameWithoutExtension(file)))
                        File.Delete(file);
                }
            }

            int index = 0;

            foreach (KeyValuePair<string, View> kvp in sqlViews)
            {
                View view = kvp.Value;

                if (view.DateLastModified > _lastUpdate)
                {
                    if (view.DateLastModified > _newLastUpdate)
                        _newLastUpdate = view.DateLastModified;

                    scripter.Options.FileName = Path.Combine(viewsPath, view.Name + ".sql");
                    scripter.Script(new Urn[] { view.Urn });
                    index++;
                }
            }

            Console.WriteLine(index);
        }

        public void ScriptStoredProcedures()
        {
            Console.Write("Scripting Stored Procedures...");

            Scripter scripter = new Scripter();
            scripter.Server = _server;
            scripter.Options.IncludeHeaders = false;
            scripter.Options.SchemaQualify = true;
            scripter.Options.SchemaQualifyForeignKeysReferences = true;
            scripter.Options.NoCollation = true;
            scripter.Options.DriAllConstraints = true;
            scripter.Options.DriAll = true;
            scripter.Options.DriAllKeys = true;
            scripter.Options.DriIndexes = true;
            scripter.Options.ClusteredIndexes = true;
            scripter.Options.NonClusteredIndexes = true;
            scripter.Options.ToFileOnly = true;

            string storedProceduresPath = Path.Combine(_path, "StoredProcedures");

            Dictionary<string, StoredProcedure> sqlStoredProcedures = new Dictionary<string, StoredProcedure>();

            foreach (StoredProcedure storedProcedure in _db.StoredProcedures)
            {
                if (!storedProcedure.IsSystemObject)
                {
                    sqlStoredProcedures.Add(storedProcedure.Name, storedProcedure);
                }
            }

            if (!Directory.Exists(storedProceduresPath))
            {
                Directory.CreateDirectory(storedProceduresPath);
            }
            else
            {
                string[] files = Directory.GetFiles(storedProceduresPath, "*.sql");

                foreach (string file in files)
                {
                    if (!sqlStoredProcedures.ContainsKey(Path.GetFileNameWithoutExtension(file)))
                        File.Delete(file);
                }
            }

            int index = 0;

            foreach (KeyValuePair<string, StoredProcedure> kvp in sqlStoredProcedures)
            {
                StoredProcedure storedProcedure = kvp.Value;

                if (storedProcedure.DateLastModified > _lastUpdate)
                {
                    if (storedProcedure.DateLastModified > _newLastUpdate)
                        _newLastUpdate = storedProcedure.DateLastModified;

                    scripter.Options.FileName = Path.Combine(storedProceduresPath, storedProcedure.Name + ".sql");
                    scripter.Script(new Urn[] { storedProcedure.Urn });
                    index++;
                }
            }

            Console.WriteLine(index);
        }

        public void ScriptUserDefinedAggregates()
        {
            Console.Write("Scripting User Defined Aggregates...");

            Scripter scripter = new Scripter();
            scripter.Server = _server;
            scripter.Options.IncludeHeaders = false;
            scripter.Options.SchemaQualify = true;
            scripter.Options.SchemaQualifyForeignKeysReferences = true;
            scripter.Options.NoCollation = true;
            scripter.Options.DriAllConstraints = true;
            scripter.Options.DriAll = true;
            scripter.Options.DriAllKeys = true;
            scripter.Options.DriIndexes = true;
            scripter.Options.ClusteredIndexes = true;
            scripter.Options.NonClusteredIndexes = true;
            scripter.Options.ToFileOnly = true;

            string userDefinedAggregatesPath = Path.Combine(_path, "UserDefinedAggregates");

            Dictionary<string, UserDefinedAggregate> sqlUDA = new Dictionary<string, UserDefinedAggregate>();

            foreach (UserDefinedAggregate uda in _db.UserDefinedAggregates)
            {
                sqlUDA.Add(uda.Name, uda);
            }

            if (!Directory.Exists(userDefinedAggregatesPath))
            {
                Directory.CreateDirectory(userDefinedAggregatesPath);
            }
            else
            {
                string[] files = Directory.GetFiles(userDefinedAggregatesPath, "*.sql");

                foreach (string file in files)
                {
                    if (!sqlUDA.ContainsKey(Path.GetFileNameWithoutExtension(file)))
                        File.Delete(file);
                }
            }

            int index = 0;


            foreach (KeyValuePair<string, UserDefinedAggregate> kvp in sqlUDA)
            {
                UserDefinedAggregate uda = kvp.Value;

                if (uda.DateLastModified > _lastUpdate)
                {
                    if (uda.DateLastModified > _newLastUpdate)
                        _newLastUpdate = uda.DateLastModified;

                    scripter.Options.FileName = Path.Combine(userDefinedAggregatesPath, uda.Name + ".sql");
                    scripter.Script(new Urn[] { uda.Urn });
                    index++;
                }
            }

            Console.WriteLine(index);
        }

        public void ScriptUserDefinedFunctions()
        {
            Console.Write("Scripting User Defined Functions...");

            Scripter scripter = new Scripter();
            scripter.Server = _server;
            scripter.Options.IncludeHeaders = false;
            scripter.Options.SchemaQualify = true;
            scripter.Options.SchemaQualifyForeignKeysReferences = true;
            scripter.Options.NoCollation = true;
            scripter.Options.DriAllConstraints = true;
            scripter.Options.DriAll = true;
            scripter.Options.DriAllKeys = true;
            scripter.Options.DriIndexes = true;
            scripter.Options.ClusteredIndexes = true;
            scripter.Options.NonClusteredIndexes = true;
            scripter.Options.ToFileOnly = true;

            string userDefinedFunctionsPath = Path.Combine(_path, "UserDefinedFunctions");

            Dictionary<string, UserDefinedFunction> sqlUDF = new Dictionary<string, UserDefinedFunction>();

            foreach (UserDefinedFunction udf in _db.UserDefinedFunctions)
            {
                if (!udf.IsSystemObject)
                {
                    sqlUDF.Add(udf.Name, udf);
                }
            }

            if (!Directory.Exists(userDefinedFunctionsPath))
            {
                Directory.CreateDirectory(userDefinedFunctionsPath);
            }
            else
            {
                string[] files = Directory.GetFiles(userDefinedFunctionsPath, "*.sql");

                foreach (string file in files)
                {
                    if (!sqlUDF.ContainsKey(Path.GetFileNameWithoutExtension(file)))
                        File.Delete(file);
                }
            }

            int index = 0;

            foreach (KeyValuePair<string, UserDefinedFunction> kvp in sqlUDF)
            {
                UserDefinedFunction udf = kvp.Value;

                if (udf.DateLastModified > _lastUpdate)
                {
                    if (udf.DateLastModified > _newLastUpdate)
                        _newLastUpdate = udf.DateLastModified;

                    scripter.Options.FileName = Path.Combine(userDefinedFunctionsPath, udf.Name + ".sql");
                    scripter.Script(new Urn[] { udf.Urn });
                    index++;
                }
            }

            Console.WriteLine(index);
        }

        public void SaveLastUpdate()
        {
            File.WriteAllText(_lastUpdateFilename, _newLastUpdate.ToBinary().ToString());
        }
    }
}
