import Topbar from './Topbar';
import React, {
  ReactElement,
  ReactNode,
  useEffect,
  useState,
  useRef,
} from 'react';
import styled from 'styled-components';
import {
  AuthService,
  EventService,
  GroupService,
  MessageService,
  Permissions,
  PublicService,
  User,
  UserService,
} from '@jbd/end-user-business-logic';
import { AuthContext, ServicesContext } from '@jbd/end-user-utils';
import { Toast } from 'primereact/toast';
import axios from 'axios';

interface Props {
  children: ReactNode;
}

const PageWrapper = styled.div`
  width: 100%;
  height: 100vh;
  display: flex;
  flex-direction: column;
  align-items: stretch;
  justify-content: flex-start;
`;

axios.defaults.baseURL = '/api';
const authService = new AuthService(axios);
const messageService = new MessageService(axios);
const eventService = new EventService(axios);
const groupService = new GroupService(axios);
const userService = new UserService(axios);
const publicService = new PublicService(axios);

const services = {
  authService,
  messageService,
  eventService,
  groupService,
  userService,
  publicService,
};

export default function AuthWrapper({ children }: Props): ReactElement {
  const [isAuthenticated, setIsAuthenticated] = useState<boolean | null>(null);
  const [user, setUser] = useState<User | null>(null);
  const [permissions, setPermissions] = useState<Permissions | null>(null);

  const toast = useRef<Toast>(null);

  const setTokenOnAxios = (t: string) => {
    axios.defaults.headers.common.Authorization = `Bearer ${t}`;
  };

  const checkValidity = async () => {
    try {
      const response = await authService.checkStatus();
      setPermissions(response.permissions);
      setUser(response.user);
      setIsAuthenticated(true);
    } catch (e) {
      toast.current.show({
        severity: 'error',
        summary: 'Sessie verlopen.',
        detail: 'Meld je opnieuw aan.',
      });
      localStorage.removeItem('token');
      setIsAuthenticated(false);
      setUser(null);
      setTokenOnAxios('undefined');
    }
  };

  useEffect(() => {
    const t = localStorage.getItem('token');
    if (t) {
      setTokenOnAxios(t);
      checkValidity();
    } else {
      setIsAuthenticated(false);
    }
  }, []);

  const login = async (
    username: string,
    password: string
  ): Promise<boolean> => {
    try {
      const {
        authToken,
        user: u,
        permissions: p,
      } = await authService.login(username, password);
      localStorage.setItem('token', authToken.token);
      setPermissions(p);
      setUser(u);
      setIsAuthenticated(true);
      setTokenOnAxios(authToken.token);
      return true;
    } catch (e) {
      console.log(e);
      setUser(null);
      setPermissions(null);
      setIsAuthenticated(false);
      setTokenOnAxios('undefined');
      return false;
    }
  };

  const logout = async () => {
    await authService.logout();
    localStorage.removeItem('token');
    setIsAuthenticated(false);
    setUser(null);
    setTokenOnAxios('undefined');
  };

  return (
    <AuthContext.Provider
      value={{
        isAuthenticated,
        user,
        permissions,
        login,
        logout,
      }}
    >
      <ServicesContext.Provider value={services}>
        <PageWrapper>
          <Topbar />
          {children}
          <Toast ref={toast} />
        </PageWrapper>
      </ServicesContext.Provider>
    </AuthContext.Provider>
  );
}
