import Link from 'next/link';
import React, { ReactElement } from 'react';
import styled from 'styled-components';
import { useRouter } from 'next/router';
import { useAuthentication } from '@jbd/end-user-utils';

const Navbar = styled.nav`
  background-color: rgb(0, 140, 214);
  display: flex;
  justify-content: space-between;
  border-bottom: 4px solid rgb(0, 112, 171);
  padding: 0.5rem 1rem;
`;

const Logo = styled.img`
  margin-top: 5px;
  max-height: 40px;
  cursor: pointer;
`;

interface TopbarButtonProps {
  title: string;
  icon: string;
  type: 'button' | 'link';
  onClick?(): void;
  to?: string;
}

const StyledTopbarLink = styled.a`
  color: white;
  text-decoration: none;

  i {
    font-weight: bold;
  }

  span {
    font-size: 85%;
    font-weight: bold;
  }

  &:hover {
    span {
      text-decoration: underline;
    }
  }
`;

function TopbarButton(props: TopbarButtonProps) {
  const { type, icon, title } = props;
  if (type === 'link') {
    const { to } = props;
    return (
      <div className="col py-0">
        <Link passHref href={to}>
          <StyledTopbarLink className="flex flex-column align-items-center justify-content-center">
            <i className={`pi pi-${icon} mb-1`} />
            <span>{title}</span>
          </StyledTopbarLink>
        </Link>
      </div>
    );
  }

  const { onClick } = props;

  return (
    <div className="col py-0">
      <StyledTopbarLink
        onClick={onClick}
        className="flex flex-column align-items-center justify-content-center cursor-pointer"
      >
        <i className={`pi pi-${icon} mb-1`} />
        <span>{title}</span>
      </StyledTopbarLink>
    </div>
  );
}

export default function Topbar(): ReactElement {
  const { isAuthenticated, logout } = useAuthentication();
  const router = useRouter();
  const isOnPublicRoute = router.pathname.includes('/publiek');

  return (
    <Navbar>
      <Link href="/" passHref>
        <Logo alt="JBD" src="/assets/jbd_logo.png" />
      </Link>
      <div className="flex flex-row justify-content-center align-items-center">
        {!isOnPublicRoute && (
          <TopbarButton
            title="Bedrijven"
            icon="briefcase"
            type="link"
            to="/publiek"
          />
        )}
        {isOnPublicRoute && isAuthenticated && (
          <TopbarButton title="Route" icon="share-alt" type="link" to="/app" />
        )}
        {isAuthenticated ? (
          <TopbarButton
            title="Afmelden"
            icon="lock"
            type="button"
            onClick={logout}
          />
        ) : (
          <TopbarButton
            title="Aanmelden"
            icon="lock-open"
            type="link"
            to="/login"
          />
        )}
      </div>
    </Navbar>
  );
}
