import React, { ReactElement } from 'react';
import { InputText } from 'primereact/inputtext';
import { InputTextarea } from 'primereact/inputtextarea';
import { Button } from 'primereact/button';
import { Toast } from 'primereact/toast';
import { useFormik } from 'formik';
import * as Yup from 'yup';
import classNames from 'classnames';
import { useQuery } from 'react-query';
import {
  useAuthentication,
  useEventContext,
  useServices,
} from '@jbd/end-user-utils';

interface Values {
  subject: string;
  body: string;
}

const validationSchema = Yup.object().shape({
  subject: Yup.string()
    .required('Onderwerp is verplicht')
    .min(3, 'Het onderwerp moet minimaal 3 karakters lang zijn.')
    .max(50, 'Het onderwerp mag maximaal 50 karakters lang zijn.'),
  body: Yup.string()
    .required('Inhoud is verplicht')
    .min(10, 'Het bericht moet minimaal 10 karakters lang zijn.')
    .max(300, 'Het bericht mag maximaal 300 karakters lang zijn.'),
});

interface Props {
  onComplete(): void;
  toast: Toast;
}

export default function MessageForm({
  onComplete,
  toast,
}: Props): ReactElement {
  const { user } = useAuthentication();
  const { group } = useEventContext();
  const { messageService } = useServices();

  const { refetch } = useQuery(
    ['user', user.id, 'messages'],
    () => messageService.getMessages(user.id),
    { enabled: false }
  );

  const formik = useFormik<Values>({
    initialValues: {
      subject: '',
      body: '',
    },
    validateOnBlur: true,
    validationSchema: validationSchema,
    onSubmit: async (data) => {
      try {
        await messageService.sendMessage(
          user.id,
          group.id,
          data.subject,
          data.body
        );
        toast.show({
          severity: 'success',
          summary: 'Bericht verzonden',
          detail: 'Het bericht is succesvol verzonden.',
        });
        refetch();
        onComplete();
      } catch (e) {
        toast.show({
          severity: 'error',
          summary: 'Er ging iets mis.',
          detail: 'Probeer het later nog eens.',
        });
      }
    },
  });

  const isFormFieldValid = (name: string) =>
    !!(formik.touched[name] && formik.errors[name]);
  const getFormErrorMessage = (name: string) => {
    return (
      isFormFieldValid(name) && (
        <small className="p-error">{formik.errors[name]}</small>
      )
    );
  };

  return (
    <div>
      <h2 className="my-0">Bericht opstellen</h2>
      <div className="pt-3 mt-3" style={{ borderTop: '1px solid #ccc' }}>
        <form
          onSubmit={formik.handleSubmit}
          className="p-fluid formgrid grid"
          style={{ maxWidth: 600 }}
        >
          <div className="p-field field col-12">
            <div>
              <label
                htmlFor="subject"
                className={classNames({
                  'p-error': isFormFieldValid('subject'),
                })}
              >
                <span className="font-bold">Onderwerp </span>
                <span
                  className={
                    formik.values.subject.length > 50
                      ? 'text-pink-600'
                      : 'text-bluegray-300'
                  }
                >
                  ({formik.values.subject.length}/50)
                </span>
              </label>
              <InputText
                id="subject"
                name="subject"
                placeholder="Onderwerp (minimaal 3 karakters)"
                value={formik.values.subject}
                onChange={formik.handleChange}
                className={classNames({
                  'mt-2': true,
                  'p-invalid': isFormFieldValid('subject'),
                })}
              />
            </div>
            {getFormErrorMessage('subject')}
          </div>
          <div className="p-field field col-12">
            <div>
              <label
                htmlFor="body"
                className={classNames({
                  'p-error': isFormFieldValid('subject'),
                })}
              >
                <span className="font-bold">Inhoud </span>
                <span
                  className={
                    formik.values.body.length > 300
                      ? 'text-pink-600'
                      : 'text-bluegray-300'
                  }
                >
                  ({formik.values.body.length}/300)
                </span>
              </label>
              <InputTextarea
                id="body"
                name="body"
                placeholder="Inhoud (minimaal 10 karakters)"
                value={formik.values.body}
                onChange={formik.handleChange}
                className={classNames({
                  'mt-2': true,
                  'p-invalid': isFormFieldValid('body'),
                })}
              />
            </div>
            {getFormErrorMessage('body')}
          </div>
          <div className="flex flex-row">
            <Button
              type="submit"
              label="Versturen"
              disabled={formik.isSubmitting || !formik.dirty || !formik.isValid}
              loading={formik.isSubmitting}
              className="my-3 mx-2"
            />
            <Button
              type="button"
              label="Annuleren"
              onClick={onComplete}
              className="my-3 mx-2 p-button-text"
            />
          </div>
        </form>
      </div>
    </div>
  );
}
