import { Message } from '@jbd/end-user-business-logic';
import React, { ReactElement } from 'react';
import { ListBox } from 'primereact/listbox';
import { Button } from 'primereact/button';
import { ScrollPanel } from 'primereact/scrollpanel';
import styled from 'styled-components';
import formatDistance from 'date-fns/formatDistance';
import nl from 'date-fns/locale/nl';

interface Props {
  messages: Message[];
  onClickCompose(): void;
  onSelectMessage(messageId: string | null): void;
  selectedMessageId: string | null;
}

const StyledScrollPanel = styled(ScrollPanel)`
  width: 300px;
  height: 400px;

  &.custom .p-scrollpanel-wrapper {
    border-right: 9px solid #f4f4f4;
  }

  &.custom .p-scrollpanel-bar {
    background-color: rgb(0, 140, 214);
    opacity: 1;
    transition: background-color 0.3s;
    border-radius: 0;
  }

  &.custom .p-scrollpanel-bar:hover {
    background-color: #135ba1;
  }

  &.custom .p-listbox .p-listbox-list {
    padding: 0;
  }

  .p-scrollpanel-content {
    height: 100%;
    overflow-y: auto;
    overflow-x: hidden;
    padding-bottom: 0;
  }
`;

function MessagesListItem({ message }: { message: Message }): ReactElement {
  return (
    <div>
      <h4 className="mt-1 mb-0">{message.sender.name}</h4>
      <span className="text-sm">{message.subject}</span>
      <div className="text-xs text-bluegray-500 mb-1">
        {formatDistance(new Date(message.receivedAt), new Date(), {
          locale: nl,
          addSuffix: true,
        })}
      </div>
    </div>
  );
}

export default function MessagesList({
  messages,
  onSelectMessage,
  onClickCompose,
  selectedMessageId,
}: Props): ReactElement {
  return (
    <>
      <Button
        label="Bericht opstellen"
        className="w-full"
        onClick={onClickCompose}
      />
      <StyledScrollPanel className="custom w-full md:w-auto">
        <ListBox
          value={messages.find((m) => m.id === selectedMessageId)}
          options={messages}
          onChange={(e) => onSelectMessage(e.value?.id ?? null)}
          itemTemplate={(option) => <MessagesListItem message={option} />}
          style={{ width: 'calc(100% + 14px)' }}
        />
      </StyledScrollPanel>
    </>
  );
}
