import { PollMetadata } from '@jbd/end-user-business-logic';
import React, { ReactElement, useRef } from 'react';
import FeedbackPollQuestion from '../molecules/FeedbackPollQuestion';
import { Button } from 'primereact/button';
import { useQuery } from 'react-query';
import Loading from '../Loading';
import { Formik, Form, FormikProps, FormikErrors } from 'formik';
import { Toast } from 'primereact/toast';
import { useAuthentication, useServices } from '@jbd/end-user-utils';

interface Props {
  poll: PollMetadata;
  toast: Toast;
  onComplete(): void;
}

type Values = Record<number, string | number | null>;

export default function FeedbackPollForm({
  poll,
  toast,
  onComplete,
}: Props): ReactElement {
  const { user } = useAuthentication();
  const { eventService } = useServices();
  const formikRef = useRef<FormikProps<Values>>(null);

  const { data } = useQuery(['user', user.id, 'polls', poll.id], () =>
    eventService.getFeedbackPollById(user.id, poll.id)
  );

  const onSubmit = async (values: Values) => {
    try {
      await eventService.submitPoll(user.id, poll.id, Object.values(values));
      toast.show({
        severity: 'success',
        summary: 'Feedback verzonden',
        detail: 'Jouw antwoorden zijn succesvol opgeslagen.',
      });
      onComplete();
    } catch (e) {
      toast.show({
        severity: 'error',
        summary: 'Er ging iets mis.',
        detail: 'Probeer het later nog eens.',
      });
    }
  };

  const onValidate = (values: Values): FormikErrors<Values> => {
    const errors = {};

    const keys = Object.keys(values);
    if (data.poll.questions.length > keys.length) {
      return {
        [data.poll.questions.length]: 'Vul alle velden in.',
      };
    }

    Object.keys(values).forEach((v) => {
      if (values[v] === null || values[v] === '') {
        errors[v] = 'Geef een antwoord alsjeblieft.';
      }
    });

    return errors;
  };

  return (
    <div>
      <h3
        className="mt-0 pb-5 text-xl"
        style={{ borderBottom: '1px solid #eee' }}
      >
        {poll.poll.title}
      </h3>
      <Formik
        onSubmit={onSubmit}
        validateOnBlur
        innerRef={formikRef}
        initialValues={{}}
        validate={onValidate}
      >
        {({ dirty, isSubmitting, isValid }) => (
          <Form>
            <div className="mt-5">
              {data ? (
                data.poll.questions.map((q, index) => (
                  <FeedbackPollQuestion
                    key={q.id}
                    question={q}
                    name={index.toString()}
                  />
                ))
              ) : (
                <Loading />
              )}
              <Button
                className="mt-2"
                disabled={!dirty || !isValid || isSubmitting}
                loading={isSubmitting}
                label="Verstuur"
                type="submit"
              />
            </div>
          </Form>
        )}
      </Formik>
    </div>
  );
}
