import { PollMetadata } from '@jbd/end-user-business-logic';
import React, { ReactElement, useState, useRef } from 'react';
import { Card } from 'primereact/card';
import { Toast } from 'primereact/toast';
import PollsList from '../molecules/PollsList';
import styled from 'styled-components';
import FeedbackPollForm from './FeedbackPollForm';
import { useQuery } from 'react-query';
import { useAuthentication, useServices } from '@jbd/end-user-utils';

interface Props {
  polls: PollMetadata[];
}

const PollsListWrapper = styled.div`
  max-width: 300px;
  @media (max-width: 767px) {
    max-width: 100%;
  }
`;

export default function FeedbackPollsContainer({ polls }: Props): ReactElement {
  const { user } = useAuthentication();
  const { eventService } = useServices();
  const { refetch } = useQuery(
    ['user', user.id, 'polls'],
    () => eventService.getFeedbackPolls(user.id),
    { enabled: false }
  );
  const [selectedPollId, setSelectedPollId] = useState<string | null>(
    polls[0].id
  );
  const selectedPoll = polls.find((m) => m.id === selectedPollId);

  const toast = useRef<Toast>(null);

  return (
    <div
      className="w-full flex flex-column md:flex-row lg:flex-row xl:flex-row justify-content-stretch align-items-stretch"
      style={{ minHeight: 300 }}
    >
      <PollsListWrapper className="flex-shrink-0">
        <PollsList
          polls={polls}
          onSelectPoll={(pollId) => {
            setSelectedPollId(pollId);
          }}
          selectedPollId={selectedPollId}
        />
      </PollsListWrapper>
      <Card className="w-full shadow-none">
        {selectedPoll ? (
          <FeedbackPollForm
            onComplete={() => {
              refetch();
              setSelectedPollId(null);
            }}
            toast={toast.current}
            poll={selectedPoll}
          />
        ) : (
          <p className="text-center text-bluegray-500">
            Selecteer een vragenlijst.
          </p>
        )}
      </Card>
      <Toast ref={toast} />
    </div>
  );
}
