import React, { ReactElement, useState, useCallback, useEffect } from 'react';
import { GoogleMap, useJsApiLoader } from '@react-google-maps/api';
import RouteSlotMarker from '../molecules/RouteSlotMarker';
import { Company, Slot } from '@jbd/end-user-business-logic';

const containerStyle = {
  width: '100%',
  height: '500px',
};

const slotHasAddress = (slot): boolean => {
  if (slot.company) {
    return slot.company.address && slot.company.address.position;
  }
  if (slot.school) {
    return slot.school.address && slot.school.address.position;
  }
  if (slot.location) {
    return slot.location.address && slot.location.address.position;
  }
  return false;
};

const getCenter = (slots: Slot[]): { lat: number; lng: number } => {
  const positions = slots
    .filter((s) => slotHasAddress(s))
    .map((s) => {
      const destination = s.company ?? s.school ?? s.location;
      return destination.address.position.coordinates;
    });
  let positions0Total = 0;
  let positions1Total = 0;
  positions.forEach((p) => {
    positions0Total += p[0];
    positions1Total += p[1];
  });
  return {
    lat: positions1Total / positions.length,
    lng: positions0Total / positions.length,
  };
};

interface Props {
  slots: Slot[];
  onSelectCompany(company: Company): void;
}

export default function RouteMap({
  slots,
  onSelectCompany,
}: Props): ReactElement {
  const { isLoaded } = useJsApiLoader({
    id: 'google-map-script',
    googleMapsApiKey: 'AIzaSyDl4siQvZqK4QUpmInE5i0v54_j9Z8Sg0s',
  });

  const [map, setMap] = useState<google.maps.Map>(null);

  const onLoad = useCallback(function callback(map) {
    const bounds = new window.google.maps.LatLngBounds();
    map.fitBounds(bounds);
    setMap(map);
  }, []);

  const onUnmount = useCallback(function callback(map) {
    setMap(null);
  }, []);

  const fitBounds = () => {
    const positions = slots
      .filter((s) => slotHasAddress(s))
      .map((s) => {
        const destination = s.company ?? s.school ?? s.location;
        return destination.address.position.coordinates;
      });

    const bounds = new window.google.maps.LatLngBounds();
    positions.forEach((p) =>
      bounds.extend(new window.google.maps.LatLng(p[1], p[0]))
    );

    if (map !== null) {
      map.fitBounds(bounds);
    }
  };

  useEffect(() => {
    if (!isLoaded || !map) {
      return;
    }
    fitBounds();
  }, [slots, isLoaded, map]);

  return (
    <div>
      {isLoaded && (
        <GoogleMap
          mapContainerStyle={containerStyle}
          center={getCenter(slots)}
          zoom={10}
          onLoad={onLoad}
          onUnmount={onUnmount}
        >
          {/* Child components, such as markers, info windows, etc. */}
          {slots
            .filter((slot) => slot.company || slot.school || slot.location)
            .map((slot) => (
              <RouteSlotMarker
                onSelectCompany={onSelectCompany}
                key={slot.id}
                slot={slot}
              />
            ))}
        </GoogleMap>
      )}
    </div>
  );
}
