import { Person } from '@jbd/end-user-business-logic';
import { Button } from 'primereact/button';
import { InputText } from 'primereact/inputtext';
import { useFormik } from 'formik';
import * as Yup from 'yup';
import { Toast } from 'primereact/toast';
import React, { ReactElement, useRef } from 'react';
import classNames from 'classnames';
import Container from '../../atoms/Container';
import { useAuthentication, useServices } from '@jbd/end-user-utils';

type Values = Partial<Person>;

const PersonalDataSchema = Yup.object().shape({
  firstName: Yup.string().nullable(),
  insertion: Yup.string().nullable(),
  lastName: Yup.string().nullable(),
  phoneNumber: Yup.string().nullable(),
});

export default function PersonalDataSection({
  person,
}: {
  person: Person;
}): ReactElement {
  const { user, logout } = useAuthentication();
  const { userService } = useServices();
  const toast = useRef<Toast>(null);

  const formik = useFormik<Values>({
    initialValues: {
      firstName: person.firstName ?? '',
      insertion: person.insertion ?? '',
      lastName: person.lastName ?? '',
      phoneNumber: person.phoneNumber ?? '',
    },
    validationSchema: PersonalDataSchema,
    onSubmit: async (data) => {
      try {
        await userService.updatePersonalData(user.id, data);
        toast.current.show({
          severity: 'success',
          summary: 'Aanpassingen opgeslagen',
          detail:
            'Het kan 15 minuten duren voordat de wijziginen zichtbaar zijn.',
        });
      } catch (e) {
        toast.current.show({
          severity: 'error',
          summary: 'Er ging iets mis.',
          detail: 'Probeer het later nog eens.',
        });
      }
    },
  });

  const isFormFieldValid = (name: string) =>
    !!(formik.touched[name] && formik.errors[name]);
  const getFormErrorMessage = (name: string) => {
    return (
      isFormFieldValid(name) && (
        <small className="p-error">{formik.errors[name]}</small>
      )
    );
  };

  return (
    <Container
      className="text-white"
      style={{ backgroundColor: 'rgb(0, 140, 214)' }}
    >
      <h1 className="my-0">Jouw gegevens</h1>
      <p className="mb-4">
        Hier kun je jouw gegevens aanpassen. Geen van de velden zijn verplicht.
      </p>
      <form
        onSubmit={formik.handleSubmit}
        className="p-fluid formgrid grid"
        style={{ maxWidth: 600 }}
      >
        <div className="p-field field col-4">
          <div>
            <label
              htmlFor="firstName"
              className={classNames({
                'font-bold': true,
                'p-error': isFormFieldValid('firstName'),
              })}
            >
              Voornaam
            </label>
            <InputText
              id="firstName"
              name="firstName"
              placeholder="Voornaam"
              value={formik.values.firstName}
              onChange={formik.handleChange}
              className={classNames({
                'mt-2': true,
                'p-invalid': isFormFieldValid('firstName'),
              })}
            />
          </div>
          {getFormErrorMessage('firstName')}
        </div>
        <div className="p-field field col-4">
          <div>
            <label
              htmlFor="insertion"
              className={classNames({
                'font-bold': true,
                'p-error': isFormFieldValid('firstName'),
              })}
            >
              Tussenvoegsel
            </label>
            <InputText
              id="insertion"
              name="insertion"
              placeholder="Tussenvoegsel"
              value={formik.values.insertion}
              onChange={formik.handleChange}
              className={classNames({
                'mt-2': true,
                'p-invalid': isFormFieldValid('insertion'),
              })}
            />
          </div>
          {getFormErrorMessage('insertion')}
        </div>
        <div className="p-field field col-4">
          <div>
            <label
              htmlFor="lastName"
              className={classNames({
                'font-bold': true,
                'p-error': isFormFieldValid('lastName'),
              })}
            >
              Achternaam
            </label>
            <InputText
              id="lastName"
              name="lastName"
              placeholder="Achternaam"
              value={formik.values.lastName}
              onChange={formik.handleChange}
              className={classNames({
                'mt-2': true,
                'p-invalid': isFormFieldValid('lastName'),
              })}
            />
          </div>
          {getFormErrorMessage('lastName')}
        </div>
        <div className="p-field field col-12">
          <span>
            <label
              htmlFor="phoneNumber"
              className={classNames({
                'text-white': true,
                'font-bold': true,
                'p-error': isFormFieldValid('phoneNumber'),
              })}
            >
              Telefoonnummer
            </label>
            <InputText
              id="phoneNumber"
              name="phoneNumber"
              placeholder="Telefoonnummer"
              value={formik.values.phoneNumber}
              onChange={formik.handleChange}
              className={classNames({
                'mt-2': true,
                'p-invalid': isFormFieldValid('phoneNumber'),
              })}
            />
          </span>
          {getFormErrorMessage('phoneNumber')}
        </div>
        <Button
          type="submit"
          label="Opslaan"
          disabled={formik.isSubmitting || !formik.dirty}
          loading={formik.isSubmitting}
          className="my-3 p-button-text bg-white"
        />
        <Button
          className="p-button-danger"
          type="button"
          onClick={logout}
          label="Uitloggen"
        />
        <Toast ref={toast} />
      </form>
    </Container>
  );
}
