import React, { ReactElement } from 'react';
import { Company } from '@jbd/end-user-business-logic';
import { useQuery } from 'react-query';
import format from 'date-fns/format';
import parseISO from 'date-fns/parseISO';
import Loading from '../../Loading';
import { Button } from 'primereact/button';
import { FontAwesomeIcon } from '@fortawesome/react-fontawesome';
import { faMapMarker, faQuestion } from '@fortawesome/free-solid-svg-icons';
import {
  VerticalTimeline,
  VerticalTimelineElement,
} from 'react-vertical-timeline-component';
import 'react-vertical-timeline-component/style.min.css';
import styled from 'styled-components';
import Container from '../../atoms/Container';
import { useEventContext, useServices } from '@jbd/end-user-utils';

const Wrapper = styled.div`
  width: 800px;
  @media (max-width: 800px) {
    width: auto;
  }

  & > p {
    text-align: center;
  }

  .vertical-timeline--two-columns
    .vertical-timeline-element.vertical-timeline-element--right
    .vertical-timeline-element-content
    .vertical-timeline-element-date,
  .vertical-timeline--two-columns
    .vertical-timeline-element:nth-child(2n):not(.vertical-timeline-element--left)
    .vertical-timeline-element-content
    .vertical-timeline-element-date {
    right: 130%;
  }

  .vertical-timeline--two-columns
    .vertical-timeline-element-content
    .vertical-timeline-element-date {
    left: 130%;
  }
`;

const Logo = styled.img`
  object-fit: cover;
  border-radius: 50%;
  width: 100%;
  height: 100%;
`;

interface Props {
  onSelectCompany(company: Company): void;
}

export default function TimelineSection({
  onSelectCompany,
}: Props): ReactElement {
  const { group, organisation, event } = useEventContext();
  const { eventService } = useServices();

  const { data } = useQuery(['group', group.id, 'route'], () =>
    eventService.getRoute(organisation.id, event.id, group.id)
  );

  if (!data) {
    return <Loading />;
  }

  const { slots } = data;

  return (
    <Container className="bg-bluegray-100">
      <h1 className="my-0">Jouw route</h1>
      <Wrapper>
        <p>
          Je gaat tijdens de Jongeren Bedrijvendag naar de volgende
          bestemmingen.
        </p>
        <VerticalTimeline>
          {slots
            ?.filter((slot) => slot.company || slot.school || slot.location)
            .map((slot) => {
              if (slot.hidden) {
                return (
                  <VerticalTimelineElement
                    key={slot.id}
                    className="vertical-timeline-element--work"
                    date={[slot.start, slot.end]
                      .map((d) => format(parseISO(d), 'HH:mm'))
                      .join(' - ')}
                    iconStyle={{
                      backgroundColor: 'rgb(0, 140, 214)',
                      color: 'white',
                    }}
                    icon={<FontAwesomeIcon icon={faQuestion} size="2x" />}
                  >
                    <h5 className="vertical-timeline-element-subtitle">
                      {slot.name}
                    </h5>
                    <p>
                      Wordt bekend gemaakt op{' '}
                      {format(parseISO(event.revealAt), 'LL')} om{' '}
                      {format(parseISO(event.revealAt), 'HH:mm ')}
                    </p>
                  </VerticalTimelineElement>
                );
              }

              if (!slot.company) {
                let destination = slot.school;
                if (!destination) {
                  destination = slot.location;
                }

                return (
                  <VerticalTimelineElement
                    key={slot.id}
                    className="vertical-timeline-element--work"
                    date={[slot.start, slot.end]
                      .map((d) => format(parseISO(d), 'HH:mm'))
                      .join(' - ')}
                    iconStyle={{
                      backgroundColor: 'rgb(0, 140, 214)',
                      color: 'white',
                    }}
                    icon={<FontAwesomeIcon icon={faMapMarker} size="2x" />}
                  >
                    <h3 className="vertical-timeline-element-title">
                      {destination.name}
                    </h3>
                    <h5 className="vertical-timeline-element-subtitle">
                      {slot.name}
                    </h5>
                    {destination.address ? (
                      <p style={{ margin: '0.25em 0 1em' }}>
                        {destination.address.street}{' '}
                        {destination.address.number}{' '}
                        {destination.address.numberAddition},
                        {destination.address.postcode}{' '}
                        {destination.address.city}
                      </p>
                    ) : null}
                  </VerticalTimelineElement>
                );
              }
              const company = slot.company;
              return (
                <VerticalTimelineElement
                  key={slot.id}
                  className="vertical-timeline-element--work"
                  date={[slot.start, slot.end]
                    .map((d) => format(parseISO(d), 'HH:mm'))
                    .join(' - ')}
                  iconStyle={{ backgroundColor: '#fff' }}
                  icon={<Logo alt={company.logo} src={company.logo} />}
                >
                  <h3 className="vertical-timeline-element-title">
                    {company.name}
                  </h3>
                  {company.address ? (
                    <p style={{ margin: '0.25em 0 1em' }}>
                      {company.address.street} {company.address.number}{' '}
                      {company.address.numberAddition},{' '}
                      {company.address.postcode} {company.address.city}
                    </p>
                  ) : null}
                  <div>
                    <Button
                      onClick={() => onSelectCompany(company)}
                      label="Bekijk bedrijf"
                    />
                  </div>
                </VerticalTimelineElement>
              );
            })}
        </VerticalTimeline>
      </Wrapper>
    </Container>
  );
}
