import React, { ReactElement } from 'react';
import styled from 'styled-components';
import { Formik, Field } from 'formik';
import { useState } from 'react';
import { InputText } from 'primereact/inputtext';
import { Button } from 'primereact/button';
import * as Yup from 'yup';
import { useRouter } from 'next/router';
import { useAuthentication } from '@jbd/end-user-utils';

const LoginContainer = styled.div`
  height: 100%;
  width: 100%;
  position: absolute;
  top: 0;
  overflow-y: auto;
  background-image: url(/assets/header1.jpg);
  background-size: cover;
  background-repeat: no-repeat;
  background-position: 50%;
  display: flex;
  flex-direction: column;
  align-items: center;
`;

const FormWrapper = styled.div`
  display: inline-block;
  position: absolute;
  top: 100px;
  padding: 20px;
  background-color: #fff;
  transition: all 0.3s ease;
  max-width: 400px;
  width: 400px;
  text-align: left;
  background-clip: initial;
  border: 1px solid rgba(0, 0, 0, 0.125);
  border-radius: 0.25rem;
`;

const Logo = styled.img`
  max-width: 200px;
`;

const LoginSchema = Yup.object().shape({
  username: Yup.string().required().min(3, 'Vul je gebruikersnaam in'),
  password: Yup.string().required().min(3, 'Vul je wachtwoord in'),
});

interface Values {
  username: string;
  password: string;
}

export default function LoginPage(): ReactElement {
  const router = useRouter();
  const { login, isAuthenticated, user, logout } = useAuthentication();
  const [isSubmitting, setIsSubmitting] = useState<boolean>(false);
  const [hasError, setHasError] = useState<boolean>(false);

  const onSubmit = async ({ username, password }: Values) => {
    setIsSubmitting(true);
    setHasError(false);
    setTimeout(async () => {
      const isAuthenticationSuccessful = await login(username, password);
      setIsSubmitting(false);
      if (isAuthenticationSuccessful) {
        setHasError(false);
        router.push('/app');
      } else {
        setHasError(true);
      }
    }, 500);
  };

  return (
    <LoginContainer>
      <FormWrapper>
        <Logo alt="Jongeren Bedrijvendag" src={'/assets/jbd_logo_blue.png'} />
        <h2>Aanmelden</h2>
        {!isAuthenticated && (
          <Formik<Values>
            onSubmit={onSubmit}
            validationSchema={LoginSchema}
            validateOnMount
            initialValues={{
              username: '',
              password: '',
            }}
          >
            {({ handleSubmit, isValid, dirty }) => (
              <form onSubmit={handleSubmit}>
                <div className="flex flex-column">
                  <label className="mt-3 mb-1" htmlFor="username">
                    <strong>Gebruikersnaam</strong>
                  </label>
                  <Field
                    as={InputText}
                    type="text"
                    id="username"
                    placeholder="Jouw gebruikersnaam"
                    name="username"
                  />

                  <label className="mt-3 mb-1" htmlFor="password">
                    <b>Wachtwoord</b>
                  </label>
                  <Field
                    as={InputText}
                    type="password"
                    id="password"
                    placeholder="Jouw wachtwoord"
                    name="password"
                  />
                  <div>
                    <Button
                      type="submit"
                      className="mt-3 mb-3 text-center"
                      disabled={!dirty || !isValid}
                      loading={isSubmitting}
                      loadingIcon="pi pi-spin pi-spinner"
                      label="Aanmelden"
                    />
                  </div>
                  {hasError && (
                    <p>
                      Er ging iets mis. Controleer je inloggegevens en probeer
                      het opnieuw. Houdt het probleem aan, neem dan contact op
                      met de systeembeheerder.
                    </p>
                  )}
                </div>
              </form>
            )}
          </Formik>
        )}
        {isAuthenticated && (
          <div>
            Je bent momenteel ingelogd met gebruikersnaam{' '}
            <strong>{user.username}</strong>. Je kunt doorgaan naar jouw
            omgeving of uitloggen.
            <p>
              <Button type="button" onClick={() => router.push('/app')}>
                Doorgaan
              </Button>
              <Button
                type="button"
                className="ml-2 p-button-secondary"
                onClick={() => logout()}
              >
                Uitloggen
              </Button>
            </p>
          </div>
        )}
        <p className="mt-4">
          Wil je de bedrijvencatalogus bekijken? Druk dan op de onderstaande
          knop.
        </p>
        <Button
          type="button"
          className="p-button-outlined"
          onClick={() => router.push('/publiek')}
        >
          Catalogus bekijken
        </Button>
      </FormWrapper>
    </LoginContainer>
  );
}
