import { Company, PublicService } from '@jbd/end-user-business-logic';
import React, { ReactElement, useEffect } from 'react';
import { useQuery } from 'react-query';
import parseISO from 'date-fns/parseISO';
import GroupSection from '../organisms/GroupSection';
import { useRouter } from 'next/router';
import Loading from '../Loading';
import FeedbackSection from '../organisms/sections/FeedbackSection';
import MessagesSection from '../organisms/sections/MessagesSection';
import PersonalDataSection from '../organisms/sections/PersonalDataSection';
import RouteMapSection from '../organisms/sections/RouteMapSection';
import SponsorsSection from '../organisms/sections/SponsorsSection';
import TimelineSection from '../organisms/sections/TimelineSection';
import {
  EventContext,
  useAuthentication,
  useServices,
} from '@jbd/end-user-utils';

export default function PrivatePage(): ReactElement {
  const { user, permissions } = useAuthentication();
  const { userService, publicService } = useServices();
  const router = useRouter();

  const { data: person, refetch: fetchPerson } = useQuery(
    ['user', user.id, 'person'],
    () => userService.getPersonalData(user.id)
  );

  const { data: organisations } = useQuery(['organisations'], () =>
    publicService.getOrganisations()
  );

  useEffect(() => {
    if (user) {
      fetchPerson();
    }
  }, [user, fetchPerson]);

  const sortedEvents = [...permissions.groups.map((g) => g.event)].sort(
    (a, b) => {
      return +parseISO(a.start) < +parseISO(b.start) ? 1 : -1;
    }
  );

  const mostRecentEvent = sortedEvents[0] ?? null;
  const group = permissions.groups.find(
    (g) => g.event.id === mostRecentEvent.id
  );
  if (!group) {
    return <Loading />;
  }
  const school = group.school;
  const organisationId = school.organisation.id;
  const organisation = organisations
    ? organisations.find((o) => o.id === organisationId)
    : school.organisation;

  const onSelectCompany = (company: Company) => {
    router.push(
      `/publiek/${organisation.slug}/bedrijven?companyId=${company.id}`
    );
  };

  return (
    <EventContext.Provider
      value={{
        event: mostRecentEvent,
        group,
        school,
        organisation,
      }}
    >
      <div>
        <RouteMapSection onSelectCompany={onSelectCompany} />
        <TimelineSection onSelectCompany={onSelectCompany} />
        <FeedbackSection />
        <MessagesSection />
        <SponsorsSection />
        <GroupSection />
        {person && <PersonalDataSection person={person} />}
      </div>
    </EventContext.Provider>
  );
}
