import { useRouter } from 'next/router';
import Head from 'next/head';
import React, { ReactElement, useEffect } from 'react';
import { useQuery } from 'react-query';
import { Organisation } from '@jbd/end-user-business-logic';
import CompanyDetailSection from '../organisms/sections/CompanyDetailSection';
import CompaniesListSection from '../organisms/sections/CompaniesListSection';
import { useServices } from '@jbd/end-user-utils';

interface Props {
  organisation: Organisation;
}

export default function PublicCompaniesPage({
  organisation,
}: Props): ReactElement {
  const router = useRouter();
  const { companyId } = router.query;
  const { publicService } = useServices();

  const { isFetched: hasFetchedCompanies, data: companies } = useQuery(
    ['public', 'organisations', organisation.id, 'companies'],
    () => publicService.getCompaniesInOrganisation(organisation.slug)
  );

  const { data: company, refetch: fetchCompany } = useQuery(
    ['public', 'organisations', organisation.id, 'companies', companyId],
    () =>
      publicService.getCompanyInOrganisationById(
        organisation.slug as string,
        companyId as string
      ),
    { enabled: false }
  );

  useEffect(() => {
    if (companyId !== undefined) {
      fetchCompany();
    }
  }, [companyId]);

  const companyFromList = companies?.find((c) => c.id === companyId);
  const combinedCompany =
    companyFromList && company ? { ...companyFromList, ...company } : undefined;

  return (
    <>
      <Head>
        <title>
          JBD &gt; {organisation.name} {company && `> ${company.name}`}
        </title>
      </Head>
      <div className="w-full">
        {combinedCompany && <CompanyDetailSection company={combinedCompany} />}
        {hasFetchedCompanies && (
          <CompaniesListSection
            companies={companies}
            categories={organisation.categories}
          />
        )}
      </div>
    </>
  );
}
