import Topbar from './Topbar';
import React, {
  ReactElement,
  ReactNode,
  useEffect,
  useState,
  useRef,
} from 'react';
import styled from 'styled-components';
import { AuthService, Permissions, User } from '@jbd/end-user-business-logic';
import { AuthContext } from '@jbd/end-user-utils';
import { Toast } from 'primereact/toast';
import axios from 'axios';
import {
  PreviewService,
  PreviewServicesContext,
} from '@jbd/preview-business-logic';

interface Props {
  children: ReactNode;
}

const PageWrapper = styled.div`
  width: 100%;
  height: 100vh;
  display: flex;
  flex-direction: column;
  align-items: stretch;
  justify-content: flex-start;
`;

axios.defaults.baseURL = process.env.NEXT_PUBLIC_JBD_API_URL + '/';
const authService = new AuthService(axios);
const previewService = new PreviewService(axios);

const services = {
  authService,
  previewService,
};

export default function AuthWrapper({ children }: Props): ReactElement {
  const [isAuthenticated, setIsAuthenticated] = useState<boolean | null>(null);
  const [user, setUser] = useState<User | null>(null);
  const [permissions, setPermissions] = useState<Permissions | null>(null);

  const toast = useRef<Toast>(null);

  const setTokenOnAxios = (t: string) => {
    axios.defaults.headers.common.Authorization = `Bearer ${t}`;
  };

  const checkValidity = async () => {
    try {
      const response = await authService.checkStatus();
      setPermissions(response.permissions);
      setUser(response.user);
      setIsAuthenticated(true);
    } catch (e) {
      toast.current.show({
        severity: 'error',
        summary: 'Sessie verlopen.',
        detail: 'Meld je opnieuw aan.',
      });
      localStorage.removeItem('authToken');
      setIsAuthenticated(false);
      setUser(null);
      setTokenOnAxios('undefined');
      if (window.location.href !== `${window.location.origin}/`) {
        setTimeout(() => {
          window.location.href = '/';
        }, 1000);
      }
    }
  };

  useEffect(() => {
    const t = localStorage.getItem('authToken');
    if (t) {
      setTokenOnAxios(JSON.parse(t).token);
      checkValidity();
    } else {
      if (window.location.href !== `${window.location.origin}/`) {
        setTimeout(() => {
          window.location.href = '/';
        }, 1000);
      }
    }
  }, []);

  const login = async (
    username: string,
    password: string
  ): Promise<boolean> => {
    try {
      const {
        authToken,
        user: u,
        permissions: p,
      } = await authService.login(username, password);
      localStorage.setItem('authToken', authToken.token);
      setPermissions(p);
      setUser(u);
      setIsAuthenticated(true);
      setTokenOnAxios(authToken.token);
      return true;
    } catch (e) {
      console.log(e);
      setUser(null);
      setPermissions(null);
      setIsAuthenticated(false);
      setTokenOnAxios('undefined');
      return false;
    }
  };

  const logout = async () => {
    await authService.logout();
    localStorage.removeItem('authToken');
    setIsAuthenticated(false);
    setUser(null);
    setTokenOnAxios('undefined');
  };

  return (
    <AuthContext.Provider
      value={{
        isAuthenticated,
        user,
        permissions,
        login,
        logout,
      }}
    >
      <PreviewServicesContext.Provider value={services}>
        <PageWrapper>
          <Topbar />
          {children}
          <Toast ref={toast} />
        </PageWrapper>
      </PreviewServicesContext.Provider>
    </AuthContext.Provider>
  );
}
