import { Category, Company } from '@jbd/end-user-business-logic';
import React, { ReactElement, useEffect, useState } from 'react';
import styled from 'styled-components';
import CompanyListItem from '../../molecules/CompanyListItem';
import { InputText } from 'primereact/inputtext';
import { Button } from 'primereact/button';
interface Props {
  companies: Company[];
  categories: Category[];
}

const SectionContainer = styled.div`
  width: 100%;
  padding: 50px;
  padding-bottom: 250px;
  background-color: #fafafa;
  color: #111;
  display: flex;
  flex-direction: column;
  align-items: center;
  justify-content: center;

  @media (max-width: 768px) {
    padding: 0;
  }
`;

export default function CompaniesListSection({
  companies,
  categories,
}: Props): ReactElement {
  const [enabledCategories, setEnabledCategories] = useState<string[]>(
    categories.map((c) => c.id)
  );

  const [filteredCompanies, setFilteredCompanies] =
    useState<Company[]>(companies);
  const [searchQuery, setSearchQuery] = useState<string>('');

  useEffect(() => {
    if (categories.length === 0) {
      setFilteredCompanies(
        companies.filter((c) =>
          c.name.toLowerCase().includes(searchQuery.trim().toLowerCase())
        )
      );
    } else {
      setFilteredCompanies(
        companies
          .filter((c) =>
            c.name.toLowerCase().includes(searchQuery.trim().toLowerCase())
          )
          .filter((c) => {
            const companyCategories = c.categories?.map((cat) => cat.id);
            return companyCategories.some((val) =>
              enabledCategories.includes(val)
            );
          })
      );
    }
  }, [searchQuery, companies, enabledCategories]);

  return (
    <SectionContainer>
      <h1 className="text-center">{companies.length} deelnemende bedrijven</h1>
      <InputText
        type="text"
        placeholder="Zoek door catalogus..."
        onChange={(e) => setSearchQuery(e.target.value)}
        value={searchQuery}
      />
      {categories.length > 0 && (
        <div style={{ maxWidth: 800 }} className="text-center mt-3">
          {categories.map((category) => (
            <Button
              className="text-white text-xs m-1 py-1 px-2 inline-block cursor-pointer font-bold border-none"
              key={category.id}
              style={{
                backgroundColor: enabledCategories.includes(category.id)
                  ? category.color
                  : '#ccc',
              }}
              label={category.name}
              onClick={() => {
                const isEnabled = enabledCategories.includes(category.id);
                if (isEnabled) {
                  setEnabledCategories(
                    enabledCategories.filter((c) => c !== category.id)
                  );
                } else {
                  setEnabledCategories([...enabledCategories, category.id]);
                }
              }}
            />
          ))}
          <Button
            className="text-white text-xs m-1 py-1 px-2 inline-block cursor-pointer font-bold border-none"
            style={{
              backgroundColor: '#777',
            }}
            label="ALLES AAN"
            onClick={() => setEnabledCategories(categories.map((c) => c.id))}
          />
          <Button
            className="text-white text-xs m-1 py-1 px-2 inline-block cursor-pointer font-bold border-none"
            style={{
              backgroundColor: '#777',
            }}
            label="ALLES UIT"
            onClick={() => setEnabledCategories([])}
          />
        </div>
      )}
      <div
        className="grid mt-4 mx-3 align-items-stretch justify-content-stretch"
        style={{ maxWidth: 900 }}
      >
        {filteredCompanies.map((company) => (
          <div
            key={company.id}
            className="sm:col-12 md:col-6 lg:col-4 xl:col-4 h-full"
          >
            <CompanyListItem company={company} />
          </div>
        ))}
      </div>
    </SectionContainer>
  );
}
