import { Company, Image } from '@jbd/end-user-business-logic';
import React, { ReactElement, useState } from 'react';
import { Card } from 'primereact/card';
import styled from 'styled-components';
import Linkify from 'react-linkify';
import { Tooltip } from 'primereact/tooltip';
import Lightbox from 'react-image-lightbox';
import 'react-image-lightbox/style.css';

interface Props {
  company: Company;
}

const Logo = styled.img`
  max-width: 300px;
  width: calc(100% - 64px);
  object-fit: contain;
  max-height: 150px;
  margin-bottom: 16px;
  margin-top: 32px;
  @media (max-width: 768px) {
    width: 100%;
  }
`;

interface SocialMediaButtonProps {
  logo: string;
  url: string;
}

function urlify(url: string) {
  return url.includes('http') ? url : 'https://' + url;
}

const SocialMediaLogo = styled.img`
  margin-right: 5px;
  width: 24px;
  height: 24px;
`;

function SocialMediaButton({ url, logo }: SocialMediaButtonProps) {
  return (
    <a href={urlify(url)} target="_blank" rel="noopener noreferrer">
      <SocialMediaLogo alt={logo} src={`/preview/assets/${logo}.png`} />
    </a>
  );
}

const SectionContainer = styled.div`
  width: 100%;
  background-color: #eee;
  color: #111;
  display: flex;
  flex-direction: row;
  align-items: center;
  justify-content: center;
  padding-top: 64px;
  padding-bottom: 64px;

  @media (max-width: 768px) {
    padding-top: 0;
    padding-bottom: 0;
  }
`;

const CardContainer = styled.div`
  width: 100%;
  display: flex;
  flex-direction: row;
  align-items: center;
  justify-content: center;
`;

const StyledLinkify = styled(Linkify)`
  color: #007bff;
  text-decoration: none;
  cursor: pointer;

  &:hover {
    text-decoration: underline;
  }
`;

const CompanyName = styled.h2`
  margin-top: 10px;
  margin-bottom: 10px;
  color: #111;
`;

const SubHeader = styled.h3`
  margin-top: 30px;
  margin-bottom: 5px;
`;

const PropertyList = styled.ul`
  list-style-type: none;
  padding-left: 0;
  & > li {
    margin-top: 2px;
    margin-bottom: 2px;
  }
`;

const StyledCard = styled(Card)`
  flex-grow: 1;
  max-width: 1200px;

  &.p-card .p-card-body,
  &.p-card .p-card-content {
    padding: 0;
  }

  @media (max-width: 768px) {
    &.p-card .p-card-body {
      max-width: 100%;
    }
  }
`;

const CardContent = styled.div`
  padding-left: 70px;
  padding-right: 70px;
  @media (max-width: 768px) {
    padding-left: 0;
    padding-right: 0;
  }
`;

const CardFooter = styled.div`
  margin-top: 30px;
  height: 8px;
  width: 100%;
  display: flex;
  flex-direction: row;
  justify-content: space-evenly;
  align-content: stretch;
`;

const FirstGalleryImage = styled.img`
  max-width: calc(100% - 20px);
  max-height: 200px;
  border-bottom-left-radius: 50%;
  border-bottom-right-radius: 50%;
  transition: all 0.25s ease;

  &:hover {
    cursor: pointer;
    filter: brightness(70%);
    border-bottom-left-radius: 0;
    border-bottom-right-radius: 0;
  }
`;

const LeftSideContainer = styled.div`
  min-width: 400px;
  @media (max-width: 420px) {
    min-width: auto;
  }

  @media (max-width: 768px) {
    padding: 10px;
  }
`;

const RightSideContainer = styled.div`
  min-width: 300px;
  @media (max-width: 420px) {
    min-width: auto;
  }
`;

function CompanyImageGallery({ images }: { images: Image[] }): ReactElement {
  const [isOpen, setIsOpen] = useState<boolean>(false);
  const [photoIndex, setPhotoIndex] = useState<number>(0);

  const firstImage = images[0];
  const currentImage = images[photoIndex];

  return (
    <div className="text-center w-full mb-3">
      <FirstGalleryImage
        alt={firstImage.filename}
        onClick={() => setIsOpen(true)}
        src={firstImage.filename}
      />
      {isOpen && (
        <Lightbox
          imageCaption={[currentImage.title, currentImage.description].join(
            ' - '
          )}
          mainSrc={currentImage.filename}
          nextSrc={images[(photoIndex + 1) % images.length].filename}
          prevSrc={
            images[(photoIndex + images.length - 1) % images.length].filename
          }
          onCloseRequest={() => setIsOpen(false)}
          onMovePrevRequest={() =>
            setPhotoIndex((photoIndex + images.length - 1) % images.length)
          }
          onMoveNextRequest={() =>
            setPhotoIndex((photoIndex + 1) % images.length)
          }
        />
      )}
    </div>
  );
}

export default function CompanyDetailSection({ company }: Props): ReactElement {
  const { address } = company;

  return (
    <SectionContainer>
      <CardContainer>
        <StyledCard>
          <CardContent>
            {company.isSponsor && (
              <div className="detail-sponsor-badge">
                <i className="pi pi-star"></i>
                <span> Sponsor</span>
              </div>
            )}
            <div
              className="flex flex-row flex-wrap lg:flex-nowrap"
              style={{ flex: '1 auto' }}
            >
              <LeftSideContainer>
                <div className="w-full text-center">
                  <Logo src={company.logo} alt={company.logo} />
                </div>
                <CompanyName>{company.name}</CompanyName>
                <PropertyList>
                  {address ? (
                    <li>
                      {address.street} {address.number}
                      {address.numberAddition
                        ? ` ${address.numberAddition}`
                        : ''}
                      , {address.postcode} {address.city}
                    </li>
                  ) : null}
                  <li>
                    <StyledLinkify as="a" properties={{ target: '_blank' }}>
                      {company.website}
                    </StyledLinkify>
                  </li>
                  <li>
                    <StyledLinkify as="a" properties={{ target: '_blank' }}>
                      {company.email}
                    </StyledLinkify>
                  </li>
                  <li>{company.phoneNumber}</li>
                  {company.employees && (
                    <li>Aantal medewerkers: {company.employees}</li>
                  )}
                  {company.sector && <li>Sector: {company.sector}</li>}
                </PropertyList>
                <div style={{ marginBottom: 20 }}>
                  {company.facebook && (
                    <SocialMediaButton
                      logo={'facebook'}
                      url={company.facebook}
                    />
                  )}
                  {company.instagram && (
                    <SocialMediaButton
                      logo={'instagram'}
                      url={company.instagram}
                    />
                  )}
                  {company.linkedin && (
                    <SocialMediaButton
                      logo={'linkedin'}
                      url={company.linkedin}
                    />
                  )}
                </div>
                <SubHeader>Profielen</SubHeader>
                <PropertyList>
                  {company.categories?.map((category) => (
                    <li
                      key={category.id}
                      style={{
                        borderLeft: `8px solid ${category.color}`,
                        fontSize: 12,
                        paddingLeft: 10,
                        marginBottom: 10,
                      }}
                    >
                      {category.name}
                    </li>
                  ))}
                </PropertyList>
                {company.recommendedCourses ? (
                  <div>
                    <SubHeader>Relevante keuzevakken</SubHeader>
                    {company.recommendedCourses.join(', ')}
                  </div>
                ) : null}
                {company.recommendedStudies ? (
                  <div>
                    <SubHeader>Vervolgopleidingen</SubHeader>
                    <ul style={{ paddingLeft: 15 }}>
                      {company.recommendedStudies.map((c) => (
                        <li key={c}>{c}</li>
                      ))}
                    </ul>
                  </div>
                ) : null}
              </LeftSideContainer>
              <RightSideContainer>
                <div>
                  {company.images?.length > 0 && (
                    <CompanyImageGallery images={company.images} />
                  )}
                  <div dangerouslySetInnerHTML={{ __html: company.info }} />
                </div>
              </RightSideContainer>
            </div>
          </CardContent>
          <CardFooter>
            {company.categories?.map((c) => (
              <div
                key={c.id}
                className="tooltippable"
                style={{ flexGrow: 1, backgroundColor: c.color, height: 8 }}
                data-pr-tooltip={c.name}
              />
            ))}
          </CardFooter>
        </StyledCard>
      </CardContainer>
      <Tooltip
        target=".tooltippable"
        mouseTrack
        mouseTrackLeft={10}
        position="top"
      />
    </SectionContainer>
  );
}
