import { useRouter } from 'next/router';
import Head from 'next/head';
import React, { ReactElement, useEffect } from 'react';
import { useQuery } from 'react-query';
import { Organisation } from '@jbd/end-user-business-logic';
import CompanyDetailSection from '../organisms/sections/CompanyDetailSection';
import CompaniesListSection from '../organisms/sections/CompaniesListSection';
import { usePreviewServices } from '@jbd/preview-business-logic';

interface Props {
  organisation: Organisation;
}

export default function PreviewCompaniesPage({
  organisation,
}: Props): ReactElement {
  const router = useRouter();
  const { companyId } = router.query;
  const { previewService } = usePreviewServices();

  const { isFetched: hasFetchedCategories, data: categories } = useQuery(
    ['preview', 'organisations', organisation.id, 'categories'],
    () => previewService.getCategoriesInOrganisation(organisation.id)
  );

  const { isFetched: hasFetchedCompanies, data: companies } = useQuery(
    ['preview', 'organisations', organisation.id, 'companies'],
    () => previewService.getCompaniesInOrganisation(organisation.id)
  );

  const { data: company, refetch: fetchCompany } = useQuery(
    ['preview', 'organisations', organisation.id, 'companies', companyId],
    () =>
      previewService.getCompanyInOrganisationById(
        organisation.id as string,
        companyId as string
      ),
    { enabled: false }
  );

  useEffect(() => {
    if (companyId !== undefined) {
      fetchCompany();
    }
  }, [companyId]);

  const companyFromList = companies?.find((c) => c.id === companyId);
  const combinedCompany =
    companyFromList && company ? { ...companyFromList, ...company } : undefined;

  return (
    <>
      <Head>
        <title>
          JBD &gt; {organisation.name} {company && `> ${company.name}`}
        </title>
      </Head>
      <div className="w-full">
        {combinedCompany && <CompanyDetailSection company={combinedCompany} />}
        {hasFetchedCompanies && (
          <CompaniesListSection companies={companies} categories={categories} />
        )}
      </div>
    </>
  );
}
