import { Contact } from "@app/shared/models/contact.model";
import { Injectable, EventEmitter } from "@angular/core";
import { HttpClient } from "@angular/common/http";

import { Company } from "@app/shared/models/company.model";
import { Observable } from "rxjs";

import { map, tap } from "rxjs/operators";

import { Member } from "@app/shared/models/member.model";
import { Organisation } from "@app/shared/models/organisation.model";
import { Image } from "@app/shared/models/image.model";
import { Category } from "@app/shared/models/category.model";
import { Slot } from "@app/shared/models/slot.model";
import { Event } from "@app/shared/models/event.model";
import { Email } from "@app/shared/emails/email.model";

export class CompanyEventInfo {
  company: Company;
  contacts: Contact[];
  organisation: Organisation;
  event: Event;
  emails: Email[];
  categories: Category[];
  slots: Slot[];
}

@Injectable()
export class CompaniesDataService {
  companiesPerOrganisation: { [organisationId: string]: Company[] } = {};
  companies: Company[];

  _currentCompany: Company = null;
  companyChanged = new EventEmitter();

  get currentCompany(): Company {
    return this._currentCompany;
  }

  set currentCompany(value: Company) {
    this._currentCompany = value;
    this.companyChanged.emit(value);
  }

  constructor(private http: HttpClient) {}

  getAll(): Observable<Company[]> {
    throw Error("check for org id");
    return this.http.get<Company[]>(`/companies`).pipe(
      map((companies: Company[]) => companies.sort((a, b) => (a.name.toLowerCase() < b.name.toLowerCase() ? -1 : 1))),
      tap(companies => (this.companies = companies))
    );
  }

  getAllForOrganisation(organisationId: string): Observable<Company[]> {
    return this.http.get<Company[]>(`/organisations/${organisationId}/companies`).pipe(
      map((companies: Company[]) => companies.sort((a, b) => (a.name.toLowerCase() < b.name.toLowerCase() ? -1 : 1))),
      tap(companies => (this.companiesPerOrganisation[organisationId] = companies))
    );
  }

  getByIdForOrganisation(organisationId: string, key: string | number): Observable<Company> {
    return this.http.get<Company>(`/organisations/${organisationId}/companies/${key}`);
  }

  getLinkedOrganisations(companyId: string): Observable<Organisation[]> {
    return this.http.get<Organisation[]>(`/companies/${companyId}/organisations`);
  }

  addToOrganisation(organisationId: string, entity: Company): Observable<Company> {
    return this.http.post<Company>(`/organisations/${organisationId}/companies`, entity);
  }

  linkToOrganisation(organisationId: string, entityId: string): Observable<Company> {
    return this.http.post<Company>(`/organisations/${organisationId}/companies`, { id: entityId });
  }

  getMembers(companyId: string): Observable<Member[]> {
    return this.http.get<Member[]>(`/companies/${companyId}/admins`);
  }

  addMember(companyId: string, username: string) {
    return this.http.post(`/companies/${companyId}/admins`, { username });
  }

  deleteMember(companyId: string, member: Member): Observable<Member> {
    return this.http.delete<Member>(`/companies/${companyId}/admins/${member.id}`);
  }

  getContacts(organisationId: string, companyId: string): Observable<Contact[]> {
    return this.http.get<Contact[]>(`/organisations/${organisationId}/companies/${companyId}/contacts`);
  }

  addContact(organisationId: string, companyId: string, contact: Contact) {
    return this.http.post<Contact>(`/organisations/${organisationId}/companies/${companyId}/contacts`, contact);
  }

  updateContact(organisationId: string, companyId: string, data: any) {
    return this.http.put<Member>(`/organisations/${organisationId}/companies/${companyId}/contacts/${data.id}`, data);
  }

  deleteContact(organisationId: string, companyId: string, contact: Contact) {
    return this.http.delete<void>(`/organisations/${organisationId}/companies/${companyId}/contacts/${contact.id}`);
  }
  update(entity: Partial<Company>): Observable<Company> {
    return this.http.put<Company>(`/companies/${entity.id}`, entity);
  }

  deleteFromOrganisation(organisationId: string, entityId: string): Observable<void> {
    return this.http.delete<void>(`/organisations/${organisationId}/companies/${entityId}`);
  }

  updateCategories(organisationId: string, entityId: string, profiles: string[]) {
    return this.http.put<Company>(`/organisations/${organisationId}/companies/${entityId}`, {
      categories: profiles.map(profile => Number.parseInt(profile))
    });
  }

  updateImages(companyId: string, images: Image[]) {
    return this.http.put(`/companies/${companyId}/images/`, images);
  }

  deleteImage(companyId: string, image: Image) {
    return this.http.request("delete", `/companies/${companyId}/images/`, { body: { imageIds: [image.id] } });
  }

  addImage(entityId: string, file: File) {
    const uploadData = new FormData();
    uploadData.append("type", "companyImages");
    uploadData.append("companyId", entityId);
    uploadData.append("files", file);

    return this.http.post("/uploads", uploadData);
  }

  getEventInfo(linkCode: string, token: string): Observable<CompanyEventInfo> {
    const t = encodeURIComponent(token);
    return this.http.get<CompanyEventInfo>(`/companies/event-info/${linkCode}?token=${t}`);
  }

  submitFeedback(linkCode: string, feedback: any): Observable<void> {
    return this.http.post<void>(`/companies/event-info/${linkCode}/emails`, feedback);
  }
}
