import { Contact } from "./../../shared/models/contact.model";
import { Component, OnDestroy, OnInit } from "@angular/core";
import * as Color from "color";
import { markFormGroupTouched } from "@app/shared/form-group";
import { FormControl, FormGroup, Validators } from "@angular/forms";
import { Company } from "@app/shared/models/company.model";
import { CompaniesDataService } from "@app/companies/companies-data.service";
import { NotificationsService } from "angular2-notifications";
import { Observable, Subscription } from "rxjs";
import { Organisation } from "@app/shared/models/organisation.model";
import { FormModalComponent, FormModalInputOptions } from "@app/shared/form-modal/form-modal.component";
import { map, tap } from "rxjs/operators";
import { NgbModal } from "@ng-bootstrap/ng-bootstrap";
import { Member } from "@app/shared/models/member.model";
import { DeleteModalComponent } from "@app/shared/delete-modal/modal.component";
import { ActivatedRoute } from "@angular/router";
import { HttpClient } from "@angular/common/http";
import { DropdownOption } from "@app/shared/forms/form-input/form-input.component";
import { CategoriesDataService } from "@app/categories/categories-data.service";
import { Category } from "@app/shared/models/category.model";
import { take } from "rxjs/internal/operators";
import { DragulaService } from "ng2-dragula";
import { Image } from "@app/shared/models/image.model";
import { Title } from "@angular/platform-browser";
import { OrganisationsDataService } from "@app/organisations/organisations-data.service";
import { environment } from "@env/environment";

const websiteReg = "(https?://)?([\\da-z.-]+)\\.([a-z.]{2,6})[/\\w .-]*/?";

@Component({
  selector: "app-company-detail",
  templateUrl: "./company-detail.component.html",
  styleUrls: ["./company-detail.component.scss"],
})
export class CompanyDetailComponent implements OnInit, OnDestroy {
  company: Company;
  ownUserId: string;
  admins: Observable<Member[]>;
  organisations: Organisation[] = [];
  info = "";
  notes = "";
  color: Color;
  config = {
    editable: true,
    spellcheck: true,
    height: "auto",
    minHeight: "200px",
    width: "auto",
    minWidth: "0",
    translate: "yes",
    enableToolbar: true,
    showToolbar: true,
    placeholder: "Enter text here...",
    imageEndPoint: "",
    toolbar: [
      ["bold", "italic", "underline", "strikeThrough", "superscript", "subscript"],
      ["justifyLeft", "justifyCenter", "justifyRight", "justifyFull", "indent", "outdent"],
      ["paragraph", "blockquote", "removeBlockquote", "horizontalLine", "orderedList", "unorderedList"],
    ],
  };
  viewFromCompany = false;
  canModify = false;

  forms = {
    logo: new FormGroup({}),
    general: new FormGroup({}),
    newCourse: new FormGroup({}),
    newStudy: new FormGroup({}),
    images: {},
  };
  subscription: Subscription;

  logoToDisplay: string | ArrayBuffer = "/assets/select-picture.png";
  logoChanged = false;
  logoFile: File = null;

  categories: Category[] = [];

  imagePreview: string | ArrayBuffer = "/assets/select-picture.png";
  imageFile: File = null;

  organisationId: string;

  contacts: Contact[] = [];
  selectedContacts: Contact[] = [];

  constructor(
    private companiesService: CompaniesDataService,
    private organisationsService: OrganisationsDataService,
    private categoriesService: CategoriesDataService,
    private toast: NotificationsService,
    private route: ActivatedRoute,
    private http: HttpClient,
    private dragulaService: DragulaService,
    private title: Title,
    private modal: NgbModal
  ) {
    this.dragulaService.createGroup("IMAGE_ORDER", {});
  }

  ngOnDestroy() {
    this.dragulaService.destroy("IMAGE_ORDER");
  }

  ngOnInit() {
    this.organisationId = this.organisationsService.currentOrganisation.id;

    this.route.params.subscribe((params) => {
      if (params.companyId) {
        this.companiesService.getByIdForOrganisation(this.organisationId, params.companyId).subscribe((company) => {
          this.initialize(company, false);
          this.categoriesService.getAllForOrganisation(this.organisationId).subscribe((categories) => {
            this.categories = categories;
          });
        });
      }
    });
  }

  initialize(company: Company, viewFromCompany: boolean) {
    this.title.setTitle(`Bedrijven > ${company.name}`);
    if (!company) {
      return;
    }

    this.viewFromCompany = viewFromCompany;

    this.company = company;
    this.color = Color("#eee");

    this.forms.general = new FormGroup({
      name: new FormControl(company.name, Validators.required),
      website: new FormControl(company.website, [Validators.pattern(websiteReg)]),
      phoneNumber: new FormControl(company.phoneNumber),
      email: new FormControl(company.email),
      postcode: new FormControl(company.address ? company.address.postcode : "", [
        Validators.pattern(/^( )*[0-9]{4}( )*[a-zA-Z]{2}( )*$/),
      ]),
      number: new FormControl(company.address ? company.address.number : "", [Validators.pattern(/^[0-9]+$/)]),
      numberAddition: new FormControl(company.address ? company.address.numberAddition : ""),
      street: new FormControl({
        value: company.address ? company.address.street : "",
        disabled: true,
      }),
      city: new FormControl({
        value: company.address ? company.address.city : "",
        disabled: true,
      }),
      facebook: new FormControl(company.facebook),
      instagram: new FormControl(company.instagram),
      linkedin: new FormControl(company.linkedin),
      allowPublic: new FormControl(company.allowPublic),
      employees: new FormControl(company.employees),
      sector: new FormControl(company.sector),
    });

    this.forms.newStudy = new FormGroup({
      name: new FormControl("", Validators.required),
    });

    this.forms.logo = new FormGroup({
      logo: new FormControl("", Validators.required),
    });

    this.forms.newCourse = new FormGroup({
      name: new FormControl("", Validators.required),
    });

    this.info = company.info;
    this.notes = company.notes;

    if (this.viewFromCompany) {
      this.canModify = true;
      this.subscription = this.companiesService.getLinkedOrganisations(this.company.id).subscribe((organisations) => {
        this.organisations = organisations;
        this.subscription.unsubscribe();
      });
    } else {
      this.canModify = this.company.canModify;
      if (!this.canModify) {
        this.forms.logo.disable();
        this.forms.general.disable();
        this.forms.newCourse.disable();
        this.forms.newStudy.disable();
      }
    }

    this.logoToDisplay = this.company.logo || "/assets/select-picture.png";

    this.getAdmins();

    this.company.images.forEach((img: Image) => {
      this.forms.images[img.id] = new FormGroup({
        id: new FormControl(img.id),
        title: new FormControl(img.title),
        description: new FormControl(img.description),
        displayOrder: new FormControl(img.displayOrder),
      });
    });

    this.companiesService
      .getContacts(this.organisationId, this.company.id)
      .subscribe((contacts: Contact[]) => (this.contacts = contacts));
  }

  getAdmins() {
    this.admins = this.companiesService
      .getMembers(this.company.id)
      .pipe(map((members) => members.filter((member) => true)));
  }

  updateGeneralInfo() {
    markFormGroupTouched(this.forms.general);
    const { number, postcode, numberAddition, name, allowPublic } = this.forms.general.value;
    const hasValidAddressOrNone =
      (postcode.length > 0 && ("" + number).length > 0) || (postcode.length === 0 && ("" + number).length === 0);

    if (this.forms.general.valid && hasValidAddressOrNone) {
      const changes = {};

      const fieldsToCheck = Object.keys(this.forms.general.value);
      const fieldsNotToCheck = ["name", "allowPublic", "number", "postcode", "numberAddition"];
      fieldsNotToCheck.forEach((element) => delete fieldsToCheck[element]);

      if (this.forms.general.get("name").dirty) {
        changes["name"] = name;
      }
      if (this.forms.general.get("allowPublic").dirty) {
        changes["allowPublic"] = allowPublic;
      }

      fieldsToCheck.forEach((element) => {
        if (this.forms.general.get(element).dirty) {
          changes[element] = this.forms.general.value[element] || null;
        }
      });

      changes["info"] = this.info;
      changes["notes"] = this.notes;
      if (this.forms.general.get("postcode").dirty || this.forms.general.get("number").dirty) {
        changes["address"] = {
          postcode: postcode,
          number: parseInt(number, 10),
          numberAddition: numberAddition || null,
        };
      }

      this.companiesService
        .update({ ...changes, id: this.company.id })
        .pipe(take(1))
        .subscribe(
          (company) => {
            this.toast.success("Aanpassing succesvol opgeslagen");
            this.company = { ...this.company, ...changes };
          },
          (err) => {
            if (err.status === 404) {
              this.toast.error("Probleem met opslaan", "Postcode en huisnummer conbinatie niet gevonden");
            } else {
              this.toast.error("Probleem met opslaan", err.message);
            }
          }
        );
    } else if (!hasValidAddressOrNone) {
      this.toast.error("Adres ongeldig");
    }
  }

  addLineToStringArray(form: FormGroup, field: string) {
    markFormGroupTouched(form);

    if (form.valid) {
      const data = [...(this.company[field] || []), form.value.name];
      this.company[field] = data;
      this.updateStringArray(data, field);
      form.reset();
    }
  }

  removeLineFromStringArray(line: string, field: string) {
    const data = this.company[field];
    data.splice(data.indexOf(line), 1);
    this.updateStringArray(data, field);
  }

  updateStringArray(data: string[], field: string) {
    this.companiesService
      .update({
        [field]: data,
        id: this.company.id,
      })
      .pipe(take(1))
      .subscribe((company) => {
        this.toast.success("Aanpassing succesvol opgeslagen");
      });
  }

  checkInputForForm(event: KeyboardEvent, form: FormGroup, field: string) {
    if (event.keyCode === 13) {
      event.preventDefault();

      this.addLineToStringArray(form, field);
    }
  }

  addAdminPressed() {
    const modalRef = this.modal.open(FormModalComponent);
    const component = modalRef.componentInstance as FormModalComponent;
    component.title = "Nieuwe beheerder toevoegen";
    component.confirmLabel = "Toevoegen";
    component.fields = [
      new FormModalInputOptions("username", "Gebruikersnaam (e-mailadres)", "text", "", [Validators.required]),
    ];
    component.callback = (result: any) => this.companiesService.addMember(this.company.id, result.username);
    component.onSuccess = (result: any) => {
      this.toast.success(`${result.username} is succesvol toegevoegd.`);
      this.admins = this.companiesService.getMembers(this.company.id).pipe(
        tap((members) => console.log(members)),
        map((members) => members.filter((member) => true))
      );
    };
  }

  deleteAdminPressed(admin: Member) {
    const modalRef = this.modal.open(DeleteModalComponent);
    const component = modalRef.componentInstance as DeleteModalComponent;
    component.name = `${admin.username} verwijderen?`;

    modalRef.result
      .then((result) => {
        this.companiesService.deleteMember(this.company.id, admin).subscribe(
          () => {
            console.log("asdf");
            this.toast.success(`${admin.username} is succesvol verwijderd.`);
            this.getAdmins();
          },
          (error) => {
            this.toast.error("Je kunt deze gebruiker niet verwijderen.");
          }
        );
      })
      .catch(() => {});
  }

  onLogoFileChanged(evt: any) {
    this.logoChanged = true;
    if (evt.target.files && evt.target.files[0]) {
      this.logoFile = evt.target.files[0];
      const reader = new FileReader();

      reader.onload = (e) => {
        this.logoToDisplay = e.target["result"];
      };

      reader.readAsDataURL(evt.target.files[0]);
    }
  }

  cancelNewLogo() {
    this.logoChanged = false;
    this.logoToDisplay = this.company.logo || "/assets/select-picture.png";
  }

  uploadNewLogo() {
    markFormGroupTouched(this.forms.logo);
    if (this.forms.logo.valid) {
      const uploadData = new FormData();
      uploadData.append("type", "companyLogo");
      uploadData.append("companyId", this.company.id);
      uploadData.append("files", this.logoFile);

      this.http.post("/uploads", uploadData).subscribe((data: { url: string }) => {
        this.toast.success("Logo succesvol opgeslagen");
        // TODO: Get Own Company
        this.logoChanged = false;
        this.logoToDisplay = data.url;
      });
    }
  }

  editCategoriesPressed() {
    const modalRef = this.modal.open(FormModalComponent);
    const component = modalRef.componentInstance as FormModalComponent;
    component.title = "Categorieën aanpassen";
    component.confirmLabel = "Aanpassen";
    component.fields = [
      new FormModalInputOptions(
        "categories",
        "Categorieën",
        "multiselect",
        {
          value: this.company.categories.map((cat) => cat.id),
          options: this.categories.map((category) => new DropdownOption(category.id, category.name)),
        },
        [Validators.required],
        {
          size: this.categories.length,
        }
      ),
    ];

    component.callback = (result: any) =>
      this.companiesService.updateCategories(this.organisationId, this.company.id, result.categories);
    component.onSuccess = () => {
      this.toast.success("Categorieén aangepast");

      this.companiesService.getByIdForOrganisation(this.organisationId, this.company.id).subscribe((company) => {
        this.initialize(company, false);
      });
    };
  }

  onImageChanged(evt: any) {
    if (evt.target.files && evt.target.files[0]) {
      this.imageFile = evt.target.files[0];
      const reader = new FileReader();

      reader.onload = (e) => {
        this.imagePreview = e.target["result"];
      };

      reader.readAsDataURL(evt.target.files[0]);
    }
  }

  uploadImage() {
    if (this.imageFile) {
      this.companiesService.addImage(this.company.id, this.imageFile).subscribe((data) => {
        this.toast.success("Afbeelding succesvol opgeslagen");
        this.companiesService.getByIdForOrganisation(this.organisationId, this.company.id).subscribe((company) => {
          this.company.images = company.images;
          this.imagePreview = "/assets/select-picture.png";
          this.imageFile = null;
          this.company.images.forEach((img: Image) => {
            this.forms.images[img.id] = new FormGroup({
              id: new FormControl(img.id),
              title: new FormControl(img.title),
              description: new FormControl(img.description),
              displayOrder: new FormControl(img.displayOrder),
            });
          });
        });
      });
    }
  }

  deleteImagePressed(image: Image) {
    const modalRef = this.modal.open(DeleteModalComponent);
    const component = modalRef.componentInstance;
    modalRef.componentInstance.name = image.title || "Deze afbeelding";

    modalRef.result
      .then((result) => {
        this.companiesService.deleteImage(this.company.id, image).subscribe((data) => {
          this.toast.success("Afbeelding verwijderd");
          this.companiesService.getByIdForOrganisation(this.organisationId, this.company.id).subscribe((company) => {
            this.company.images = company.images;
          });
        });
      })
      .catch(() => {});
  }

  imagesSortOrderChanged(images: Image[]) {
    for (let i = 0; i < images.length; ++i) {
      this.forms.images[images[i].id].get("displayOrder").setValue(i);
      this.forms.images[images[i].id].markAsDirty();
    }
  }

  updateImages() {
    const values = Object.values(this.forms.images).map((formGroup: FormGroup) => formGroup.value);
    this.companiesService
      .updateImages(this.company.id, values)
      .subscribe(() => this.toast.success("Aanpassingen opgeslagen"));
  }

  contactClicked(event: any) {
    const contact = event.selected[0] as Contact;

    const modalRef = this.modal.open(FormModalComponent);
    const component = modalRef.componentInstance as FormModalComponent;
    component.title = "Contact aanpassen";
    component.confirmLabel = "Aanpassen";
    component.fields = [
      new FormModalInputOptions("title", "Titel", "text", contact.title),
      new FormModalInputOptions("firstName", "Voornaam", "text", contact.firstName),
      new FormModalInputOptions("insertion", "Tussenvoegsel", "text", contact.insertion),
      new FormModalInputOptions("lastName", "Achternaam", "text", contact.lastName),
      new FormModalInputOptions("phoneNumber", "Telefoonnummer", "text", contact.phoneNumber),
      new FormModalInputOptions("email", "E-mailadres", "text", contact.email),
      new FormModalInputOptions("receiveEmail", "Ontvangt mails", "switch", contact.receiveEmail),
      new FormModalInputOptions("notes", "Notities", "textarea", contact.notes),
    ];

    component.callback = (result: any) => {
      return this.companiesService.updateContact(this.organisationId, this.company.id, {
        ...contact,
        ...result,
        id: contact.id,
      });
    };
    component.onSuccess = () => {
      this.companiesService.getByIdForOrganisation(this.organisationId, this.company.id).subscribe((company) => {
        this.initialize(company, false);
      });
    };
    component.hasDeleteCallback = true;
    component.deleteCallback = () => {
      this.companiesService.deleteContact(this.organisationId, this.company.id, contact).subscribe(() => {
        this.toast.success("Contact verwijderd.");
        this.companiesService
          .getContacts(this.organisationId, this.company.id)
          .subscribe((contacts: Contact[]) => (this.contacts = contacts));
      });
    };
  }

  newPressed() {
    const modalRef = this.modal.open(FormModalComponent);
    const component = modalRef.componentInstance as FormModalComponent;
    component.callback = (result) => this.companiesService.addContact(this.organisationId, this.company.id, result);
    component.onSuccess = () => {
      this.companiesService.getByIdForOrganisation(this.organisationId, this.company.id).subscribe((company) => {
        this.initialize(company, false);
      });
    };
    component.title = "Nieuw contact toevoegen";
    component.confirmLabel = "Toevoegen";
    component.fields = [
      new FormModalInputOptions("title", "Titel", "text", ""),
      new FormModalInputOptions("firstName", "Voornaam", "text", ""),
      new FormModalInputOptions("insertion", "Tussenvoegsel", "text", ""),
      new FormModalInputOptions("lastName", "Achternaam", "text", ""),
      new FormModalInputOptions("phoneNumber", "Telefoonnummer", "text", ""),
      new FormModalInputOptions("email", "E-mailadres", "text", ""),
      new FormModalInputOptions("receiveEmail", "Ontvangt mails", "switch", true),
      new FormModalInputOptions("notes", "Notities", "textarea", ""),
    ];
  }

  showPreview(company: Company) {
    const organisation = this.organisationsService._currentOrganisation;
    window.open(
      `https://portal.jongerenbedrijvendag.nl/preview/?organisationId=${organisation.id}&companyId=${company.id}`
    );
  }
}
