import { markFormGroupTouched } from "@app/shared/form-group";
import { FormGroup, FormControl, Validators } from "@angular/forms";
import { Contact } from "@app/shared/models/contact.model";
import { Component, OnInit } from "@angular/core";
import * as Color from "color";
import { Company } from "@app/shared/models/company.model";
import { CompaniesDataService } from "@app/companies/companies-data.service";
import { Organisation } from "@app/shared/models/organisation.model";
import { ActivatedRoute } from "@angular/router";
import { Category } from "@app/shared/models/category.model";
import { Title } from "@angular/platform-browser";
import { Event } from "@app/shared/models/event.model";
import { Slot } from "@app/shared/models/slot.model";
import { Image } from "@app/shared/models/image.model";
import { Group } from "@app/shared/models/group.model";
import { Lightbox, LightboxConfig } from "ngx-lightbox";
import { NotificationsService } from "angular2-notifications";
import { Email } from "@app/shared/emails/email.model";
import * as Markdown from "markdown-it";
import { DropdownOption } from "@app/shared/forms/form-input/form-input.component";

@Component({
  selector: "app-companies-unauthed-detail",
  templateUrl: "./companies-unauthed-detail.component.html",
  styleUrls: ["./companies-unauthed-detail.component.scss"]
})
export class CompaniesUnauthedDetailComponent implements OnInit {
  company: Company;
  contacts: Contact[];
  event: Event;
  organisation: Organisation;
  slots: Slot[];
  emails: Email[];
  categories: Category[];
  feedbackForm: FormGroup;
  sending = false;
  md: Markdown;
  color: Color;
  contactOptions: DropdownOption[] = [];

  token = "";
  id = "";

  constructor(
    private companiesService: CompaniesDataService,
    private route: ActivatedRoute,
    private toast: NotificationsService,
    private title: Title,
    private lightbox: Lightbox,
    private lighboxConfig: LightboxConfig
  ) {
    lighboxConfig.fadeDuration = 0.4;
    this.md = new Markdown({ html: false });
  }

  ngOnInit() {
    this.route.params.subscribe(params => {
      this.token = params.token;
      this.id = params.id;
      this.initialize();
    });
  }

  initialize() {
    const { contactName = "", contactEmail = "" } = this.route.snapshot.queryParams;

    this.companiesService.getEventInfo(this.id, this.token).subscribe(result => {
      this.organisation = result.organisation;
      this.color = result.organisation.primaryColor;
      this.event = result.event;
      this.slots = result.slots;
      this.categories = result.categories;
      this.company = result.company;
      this.emails = result.emails;

      this.contacts = result.contacts.map(c => {
        const contact = new Contact();
        Object.keys(c).forEach(k => (contact[k] = c[k]));
        this.contactOptions.push(new DropdownOption(contact.email, `${contact.name} <${contact.email}>`));
        return contact;
      });
      const selectedContact = this.contacts.find(c => c.email === contactEmail) || this.contacts[0];

      this.feedbackForm = new FormGroup({
        contact: new FormControl(selectedContact.email),
        contactName: new FormControl(contactName, Validators.required),
        contactEmail: new FormControl(contactEmail, Validators.required),
        subject: new FormControl(`Re: ${this.emails[this.emails.length - 1].subject}`, Validators.required),
        feedback: new FormControl("", Validators.required)
      });

      this.title.setTitle(this.company.name + " > " + result.event.name);
    });
  }

  getImageTooltip(image: Image) {
    return [image.title, image.description].filter(s => s && s.length > 0).join(" - ");
  }

  getCategoriesAsString(categories: Category[]) {
    return categories.map(cat => cat.name).join(", ") || "Onbekend";
  }

  getMarkdownParsedValue(body: string) {
    return this.md.render(body);
  }

  getDietInfo(group: Group) {
    const dietInfo = group.members.map(member => (member.person ? member.person.dietInfo : null));

    const counts = {};
    dietInfo.forEach(x => {
      counts[x] = (counts[x] || 0) + 1;
    });
    delete counts["null"];

    if (Object.keys(counts).length === 0) {
      return "Geen dieetwensen";
    }

    return (
      "Dieetwensen: " +
      Object.keys(counts)
        .map(countKey => `${countKey}: ${counts[countKey]}x`)
        .join(", ")
    );
  }

  openImage(index: number): void {
    // open lightbox
    this.lightbox.open(
      this.company.images.map(image => {
        return {
          src: image.filename,
          caption: [image.title, image.description].filter(s => s && s.length > 0).join(" - "),
          thumb: image.filename
        };
      }),
      index
    );
  }

  onSubmitFeedback() {
    markFormGroupTouched(this.feedbackForm);

    if (this.feedbackForm.valid) {
      const values = this.feedbackForm.value;

      const data = {
        eventToken: this.token,
        emailId: this.emails[this.emails.length - 1].id,
        senderEmail: values.contact,
        senderName: this.contacts.find(c => c.email === values.contact).name,
        recipientEmail: this.emails[0].senderEmail,
        recipientName: this.emails[0].senderName,
        subject: values.subject,
        body: values.feedback
      };

      this.companiesService.submitFeedback(this.id, data).subscribe(() => {
        this.toast.success("Feedback verzonden.");
        this.feedbackForm.reset();
        this.initialize();
      });
    }
  }
}
