import { Injectable } from "@angular/core";
import { HttpEvent, HttpInterceptor, HttpHandler, HttpRequest } from "@angular/common/http";
import { Observable } from "rxjs";
import { catchError } from "rxjs/operators";

import { environment } from "@env/environment";
import { Logger } from "../logger.service";
import { AuthDataService } from "../../shared/auth-data.service";

import * as Sentry from "@sentry/angular";

const log = new Logger("ErrorHandlerInterceptor");

/**
 * Adds a default error handler to all requests.
 */
@Injectable()
export class ErrorHandlerInterceptor implements HttpInterceptor {
  constructor(private authDataService: AuthDataService) {}

  intercept(request: HttpRequest<any>, next: HttpHandler): Observable<HttpEvent<any>> {
    return next.handle(request).pipe(catchError((error) => this.errorHandler(error, request)));
  }

  // Customize the default error handler here if needed
  private errorHandler(response: HttpEvent<any>, request: HttpRequest<any>): Observable<HttpEvent<any>> {
    if (response["status"] === 401) {
      this.authDataService.triggerLogout.emit();
    }

    if (!environment.production) {
      // Do something with the error
      log.error("Request error", response);
    }

    Sentry.captureException({
      request: request.body,
      response,
    });

    throw response;
  }
}
