import { Component, OnInit } from "@angular/core";
import { ActivatedRoute, NavigationEnd, Route, Router } from "@angular/router";
import * as moment from "moment";
import { HttpCacheService } from "@app/core/http/http-cache.service";
import { filter, map } from "rxjs/operators";
import { merge, of } from "rxjs";
import { OrganisationsDataService } from "@app/organisations/organisations-data.service";
import { Organisation } from "@app/shared/models/organisation.model";
import { Event } from "@app/shared/models/event.model";
import { School } from "@app/shared/models/school.model";
import { SchoolsDataService } from "@app/schools/schools-data.service";
import { EventsDataService } from "@app/events/events-data.service";
import { Company } from "@app/shared/models/company.model";
import { CompaniesDataService } from "@app/companies/companies-data.service";
import { AuthService } from '../../shared/auth.service';
import { AuthDataService } from '../../shared/auth-data.service';

export class HeaderOption {
  title: string;
  icon: string;
  link: string;
  isDivider?: boolean;
}

@Component({
  selector: "app-shell",
  templateUrl: "./shell.component.html",
  styleUrls: ["./shell.component.scss"]
})
export class ShellComponent implements OnInit {
  options: HeaderOption[] = [];
  mainRoute = "";
  routes: Route[];

  organisation: Organisation = null;
  event: Event = null;
  school: School = null;
  company: Company = null;

  constructor(
    private httpCacheService: HttpCacheService,
    private organisationsService: OrganisationsDataService,
    private eventsService: EventsDataService,
    private schoolsService: SchoolsDataService,
    private companiesService: CompaniesDataService,
    private authService: AuthService,
    private authDataService: AuthDataService,
    private route: ActivatedRoute,
    private router: Router
  ) {}

  ngOnInit() {
    merge(
      this.router.events.pipe(
        filter(event => event instanceof NavigationEnd),
        map(() => this.route)
      ),
      of(this.route)
    )
      .pipe(
        map((route: ActivatedRoute) => {
          const routes = [];
          routes.push(route.routeConfig);
          let lastRoute = route;
          while (lastRoute.firstChild) {
            routes.push(lastRoute.firstChild.routeConfig);
            lastRoute = lastRoute.firstChild;
          }
          return routes;
        })
      )
      .subscribe(routes => {
        this.routes = routes.slice(1);
        this.updateNavigation();
      });

    this.organisationsService.organisationChanged.subscribe((organisation: Organisation) => {
      this.organisation = organisation;
      this.updateNavigation();
    });
    this.schoolsService.schoolChanged.subscribe((school: School) => {
      this.school = school;
      this.updateNavigation();
    });
    this.companiesService.companyChanged.subscribe((company: Company) => {
      this.company = company;
      this.updateNavigation();
    });
    this.eventsService.eventChanged.subscribe((event: Event) => {
      this.event = event;
      this.updateNavigation();
    });

    this.httpCacheService.cleanCache(
      moment()
        .subtract(3, "minutes")
        .toDate()
    );

    const authToken = window.localStorage.getItem("authToken");
    if (authToken) {
      const expirationDate = JSON.parse(authToken).expiresAt;

      if (moment(expirationDate).isBefore(Date.now())) {
        window.localStorage.setItem("sessionExpired", "true");
        this.authService.logout();
      } else {
        this.authService.getStatus();
        this.init();
      }
    }
  }

  init() {
    const permissions: any = JSON.parse(window.localStorage.getItem("permissions")) || null;
    this.authDataService.permissions = permissions;
  }

  updateNavigation() {
    const mainRoute = this.routes[0].path;
    switch (mainRoute) {
      case "account":
        this.options = [
          { title: "Uw account", icon: "user-circle", link: "/account" },
          { title: "Terug", icon: "arrow-left", link: "/" }
        ];
        break;
      case "organisaties":
        this.options = [{ title: "Uw organisaties", icon: "sitemap", link: "/organisaties" }];

        if (this.organisation && this.routes.length > 1 && this.routes[1].path === ":organisationId") {
          this.options = this.options.concat([
            { icon: "", isDivider: true, link: "", title: this.organisation.name },
            {
              title: "Evenementen",
              icon: "calendar",
              link: `/organisaties/${this.organisation.id}/evenementen`
            },
            {
              title: "Scholen",
              icon: "school",
              link: `/organisaties/${this.organisation.id}/scholen`
            },
            {
              title: "Bedrijven",
              icon: "building",
              link: `/organisaties/${this.organisation.id}/bedrijven`
            },
            {
              title: "Push-berichten",
              icon: "bell",
              link: `/organisaties/${this.organisation.id}/push-berichten`
            },
            {
              title: "Communicatie",
              icon: "comments",
              link: `/organisaties/${this.organisation.id}/communicatie`
            },
            {
              title: "Templates",
              icon: "newspaper",
              link: `/organisaties/${this.organisation.id}/templates`
            },
            {
              title: "Accounts",
              icon: "user-cog",
              link: `/organisaties/${this.organisation.id}/accounts`
            },
            {
              title: "Categorieën",
              icon: "th-list",
              link: `/organisaties/${this.organisation.id}/categorieen`
            },
            {
              title: "Locaties",
              icon: "map-marker",
              link: `/organisaties/${this.organisation.id}/locaties`
            }
          ]);

          if (this.school && this.routes.length > 3 && this.routes[3].path === ":schoolId") {
            this.options = this.options.concat([
              { icon: "", isDivider: true, link: "", title: this.school.name },
              {
                title: "School aanpassen",
                icon: "edit",
                link: `/organisaties/${this.organisation.id}/scholen/${this.school.id}`
              },
              {
                title: "Communicatie",
                icon: "comments",
                link: `/organisaties/${this.organisation.id}/scholen/${this.school.id}/communicatie`
              },
              {
                title: "Groepen",
                icon: "users",
                link: `/organisaties/${this.organisation.id}/scholen/${this.school.id}/groepen`
              },
              {
                title: "Accounts",
                icon: "user-cog",
                link: `/organisaties/${this.organisation.id}/scholen/${this.school.id}/accounts`
              },
              {
                title: "Leerlingen importeren",
                icon: "upload",
                link: `/organisaties/${this.organisation.id}/scholen/${this.school.id}/importeren`
              },
              {
                title: "Geïmporteerde batches",
                icon: "th",
                link: `/organisaties/${this.organisation.id}/scholen/${this.school.id}/batches`
              }
            ]);
          }

          if (this.company && this.routes.length > 3 && this.routes[3].path === ":companyId") {
            this.options = this.options.concat([
              { icon: "", isDivider: true, link: "", title: this.company.name },
              {
                title: "Bedrijf aanpassen",
                icon: "edit",
                link: `/organisaties/${this.organisation.id}/bedrijven/${this.company.id}`
              },
              {
                title: "Communicatie",
                icon: "comments",
                link: `/organisaties/${this.organisation.id}/bedrijven/${this.company.id}/communicatie`
              }
            ]);
          }

          if (this.event && this.routes.length > 3 && this.routes[3].path === ":eventId") {
            this.options = this.options.concat([
              { icon: "", isDivider: true, link: "", title: this.event.name },
              {
                title: "Evenement aanpassen",
                icon: "edit",
                link: `/organisaties/${this.organisation.id}/evenementen/${this.event.id}`
              },
              {
                title: "Deelnemende bedrijven",
                icon: "building",
                link: `/organisaties/${this.organisation.id}/evenementen/${this.event.id}/bedrijven`
              },
              {
                title: "Routeplanning",
                icon: "calendar-alt",
                link: `/organisaties/${this.organisation.id}/evenementen/${this.event.id}/planning`
              }
            ]);
          }
          break;
        }
    }
  }
}
