import { Component, OnDestroy, OnInit } from "@angular/core";
import { forkJoin } from "rxjs";
import { EventsDataService } from "@app/events/events-data.service";
import { Event } from "@app/shared/models/event.model";
import { FormGroup, FormBuilder } from "@angular/forms";
import { markFormGroupTouched } from "app/shared/form-group";
import { NotificationsService } from "angular2-notifications";
import { NgbModal } from "@ng-bootstrap/ng-bootstrap";
import { CrudComponent } from "@app/shared/crud.component";
import { Company } from "@app/shared/models/company.model";
import { CompaniesDataService } from "@app/companies/companies-data.service";
import { Title } from "@angular/platform-browser";
import { FormModalComponent } from "@app/shared/form-modal/form-modal.component";
import { Organisation } from "@app/shared/models/organisation.model";
import { OrganisationsDataService } from "@app/organisations/organisations-data.service";

@Component({
  selector: "app-event-companies",
  styleUrls: ["event-companies.component.scss"],
  templateUrl: "event-companies.component.html"
})
export class EventCompaniesComponent extends CrudComponent<Event> implements OnInit {
  // New
  event: Event = null;
  organisation: Organisation = null;

  forms = {
    companies: new FormGroup({}),
    sponsors: new FormGroup({})
  };

  organisationCompanies: Company[] = [];
  eventCompanies: Company[] = [];
  eventSponsors: Company[] = [];
  filteredCompanies: Company[] = [];

  constructor(
    private organisationsService: OrganisationsDataService,
    private eventsService: EventsDataService,
    private toast: NotificationsService,
    private formBuilder: FormBuilder,
    private companiesService: CompaniesDataService,
    private title: Title,
    public modalService: NgbModal
  ) {
    super();
    super.initialize(modalService);
  }

  ngOnInit() {
    this.organisation = this.organisationsService.currentOrganisation;
    this.event = this.eventsService.currentEvent;

    this.companiesService.getAllForOrganisation(this.organisation.id).subscribe(companies => {
      this.organisationCompanies = companies;
      this.filteredCompanies = companies;
      this.initializeCompanies();
    });

    this.title.setTitle(`Organisaties > ${this.organisation.name} > Evenementen > ${this.event.name} > Bedrijven`);
  }

  initializeCompanies() {
    this.eventsService.getCompaniesForId(this.organisation.id, this.event.id).subscribe(companies => {
      this.eventCompanies = companies;

      this.forms.companies = this.formBuilder.group({
        companies: this.buildForm(this.eventCompanies)
      });

      setTimeout(() => {
        if (this.table) {
          this.table.messages.totalMessage = "bedrijven";
        }
      }, 500);
    });

    this.eventsService.getCompaniesForId(this.organisation.id, this.event.id, "sponsors").subscribe(sponsors => {
      this.eventSponsors = sponsors;

      this.forms.sponsors = this.formBuilder.group({
        companies: this.buildForm(this.eventSponsors)
      });
    });
  }

  buildForm(companies: Company[]) {
    return this.formBuilder.array(
      this.organisationCompanies.map(oc => {
        return this.formBuilder.control(companies.findIndex(ec => ec.id === oc.id) !== -1);
      })
    );
  }

  onSubmitCompanies() {
    markFormGroupTouched(this.forms.companies);

    const values = this.forms.companies.value.companies;
    const remove = this.organisationCompanies
      .filter((oc, i) => this.eventCompanies.findIndex(ec => ec.id === oc.id) !== -1 && !values[i])
      .map(c => c.id);
    const add = this.organisationCompanies
      .filter((oc, i) => this.eventCompanies.findIndex(ec => ec.id === oc.id) === -1 && values[i])
      .map(c => c.id);

    const parallel = [];
    if (remove.length > 0) {
      parallel.push(this.eventsService.removeCompaniesForId(this.organisation.id, this.event.id, remove));
    }

    if (add.length > 0) {
      parallel.push(this.eventsService.addCompaniesForId(this.organisation.id, this.event.id, add));
    }

    if (parallel.length > 0) {
      forkJoin(parallel).subscribe(() => {
        this.eventsService.getCompaniesForId(this.organisation.id, this.event.id).subscribe(companies => {
          this.eventCompanies = companies;
          this.toast.success("Deelnemende bedrijven opgeslagen.");
        });
      });
    }

    this.onSubmitSponsors();
  }

  onSubmitSponsors() {
    markFormGroupTouched(this.forms.sponsors);

    const values = this.forms.sponsors.value.companies;
    const remove = this.organisationCompanies
      .filter((oc, i) => this.eventSponsors.findIndex(ec => ec.id === oc.id) !== -1 && !values[i])
      .map(c => c.id);
    const add = this.organisationCompanies
      .filter((oc, i) => this.eventSponsors.findIndex(ec => ec.id === oc.id) === -1 && values[i])
      .map(c => c.id);

    const parallel = [];
    if (remove.length > 0) {
      parallel.push(this.eventsService.removeCompaniesForId(this.organisation.id, this.event.id, remove, "sponsors"));
    }

    if (add.length > 0) {
      parallel.push(this.eventsService.addCompaniesForId(this.organisation.id, this.event.id, add, "sponsors"));
    }

    if (parallel.length > 0) {
      forkJoin(parallel).subscribe(() => {
        this.eventsService.getCompaniesForId(this.organisation.id, this.event.id, "sponsors").subscribe(sponsors => {
          this.eventSponsors = sponsors;
          this.toast.success("Sponsoren opgeslagen.");
        });
      });
    }
  }

  updateCompaniesFilter(event: any) {
    const val = event.target.value.toLowerCase();

    // filter our data
    const temp = this.organisationCompanies.filter(d => {
      return d.name.toLowerCase().indexOf(val) !== -1 || !val;
    });

    // update the rows
    this.filteredCompanies = temp;
    // Whenever the filter changes, always go back to the first page
    this.table.offset = 0;
  }

  rowClicked(event: any) {
    const control = (this.forms.companies.get("companies") as FormGroup).controls[
      this.organisationCompanies.findIndex(oc => oc.id === event.selected[0].id)
    ];
    control.setValue(!control.value);
  }

  findIndexForCompany(company: Company) {
    return this.organisationCompanies.findIndex(oc => oc.id === company.id);
  }

  mailOpened(company: Company): boolean {
    const eventCompany = this.eventCompanies.find(ec => ec.id === company.id);
    return eventCompany && eventCompany.mailOpened;
  }

  mailSent(company: Company): boolean {
    const eventCompany = this.eventCompanies.find(ec => ec.id === company.id);
    return eventCompany && eventCompany.mailSentOn !== null;
  }

  onSendMails() {
    const modalRef = this.modalService.open(FormModalComponent);
    const component = modalRef.componentInstance as FormModalComponent;
    component.title = "Mails versturen";
    component.confirmLabel = "Versturen";
    component.description = "Wilt u mails versturen naar alle bedrijven waar nog geen mail naar verstuurd is?";

    component.callback = result => {
      return this.eventsService.mailCompanies(this.organisation.id, this.event.id);
    };

    component.onSuccess = () => {
      this.toast.success("Mails verstuurd.");

      this.eventsService.getCompaniesForId(this.organisation.id, this.event.id).subscribe(companies => {
        this.eventCompanies = companies;
      });
    };
  }

  onSendMail(company: Company) {
    const modalRef = this.modalService.open(FormModalComponent);
    const component = modalRef.componentInstance as FormModalComponent;
    component.title = "Mails versturen";
    component.confirmLabel = "Versturen";
    component.description = `Wilt u een mail versturen naar ${company.name}?`;
    if (company.mailSentOn !== null && company.mailOpened === false) {
      component.description += " Naar dit bedrijf is al een mail verstuurd.";
    } else if (company.mailSentOn !== null && company.mailOpened) {
      component.description += " Dit bedrijf heeft de mail al bekeken.";
    }

    component.callback = result => {
      return this.eventsService.mailCompanies(this.organisation.id, this.event.id, company.id);
    };

    component.onSuccess = () => {
      this.toast.success("Mail verstuurd.");
    };
  }
}
