import { Component, OnInit } from "@angular/core";
import * as Color from "color";
import { first, take } from "rxjs/internal/operators";
import { EventsDataService } from "@app/events/events-data.service";
import { Event, travelModeOptions } from "@app/shared/models/event.model";
import { PredefinedSlot } from "@app/shared/models/predefined-slot.model";
import * as moment from "moment";
import { Router } from "@angular/router";
import { FormGroup, FormControl, Validators } from "@angular/forms";
import { markFormGroupTouched } from "app/shared/form-group";
import { NotificationsService } from "angular2-notifications";
import { NgbModal } from "@ng-bootstrap/ng-bootstrap";
import { FormModalComponent, FormModalInputOptions } from "app/shared/form-modal/form-modal.component";
import { DropdownOption } from "app/shared/forms/form-input/form-input.component";
import { DateValidators } from "app/shared/validators/date.validator";
import { DeleteModalComponent } from "app/shared/delete-modal/modal.component";
import { CrudComponent } from "@app/shared/crud.component";
import { Title } from "@angular/platform-browser";
import { Organisation } from "@app/shared/models/organisation.model";
import { OrganisationsDataService } from "@app/organisations/organisations-data.service";

@Component({
  selector: "app-organisation-event-detail",
  styleUrls: ["event-detail.component.scss"],
  templateUrl: "event-detail.component.html",
})
export class EventDetailComponent extends CrudComponent<Event> implements OnInit {
  // New
  event: Event = null;
  organisation: Organisation = null;
  color: Color;

  forms = {
    info: new FormGroup({}),
    companies: new FormGroup({}),
    sponsors: new FormGroup({}),
  };

  predefinedSlots: PredefinedSlot[];

  cardVisibility = {
    general: true,
    slots: true,
  };

  constructor(
    private eventsService: EventsDataService,
    private organisationsService: OrganisationsDataService,
    private toast: NotificationsService,
    private router: Router,
    private title: Title,
    public modalService: NgbModal
  ) {
    super();
    super.initialize(modalService);
  }

  ngOnInit() {
    this.predefinedSlots = [];
    this.organisation = this.organisationsService.currentOrganisation;

    this.event = this.eventsService.currentEvent;

    this.title.setTitle(`Organisaties > ${this.organisation.name} > Evenementen > ${this.event.name}`);
    this.createForm();
    this.eventsService.getPredefinedSlotsForId(this.organisation.id, this.event.id).subscribe((slots) => {
      this.predefinedSlots = slots;
    });
  }

  getPrimaryTravelModes() {
    return travelModeOptions;
  }

  createForm() {
    if (this.event !== null) {
      this.color = Color("#eee");

      this.forms.info = new FormGroup({
        name: new FormControl("", Validators.required),
        start: new FormControl("", [Validators.required, DateValidators.isValidFormat]),
        end: new FormControl("", [Validators.required, DateValidators.isValidFormat]),
        publishAt: new FormControl("", [DateValidators.isValidFormat]),
        revealAt: new FormControl("", [DateValidators.isValidFormat]),
        primaryTravelMode: new FormControl("", Validators.required),
      });

      this.resetForm(this.event);
    }
  }

  resetForm(event: Event) {
    this.forms.info.reset();
    setTimeout(() => {
      this.forms.info.reset({
        name: event.name,
        start: moment(event.start).toDate(),
        end: moment(event.end).toDate(),
        publishAt: (event.publishAt && moment(event.publishAt).toDate()) || "",
        revealAt: (event.revealAt && moment(event.revealAt).toDate()) || "",
        primaryTravelMode: event.primaryTravelMode,
      });
    });
  }

  deletePressed() {
    const modalRef = this.modalService.open(DeleteModalComponent);
    const component = modalRef.componentInstance;
    component.name = this.event.name;

    modalRef.result
      .then((result) => {
        this.eventsService.deleteFromOrganisation(this.organisation.id, this.event.id).subscribe(() => {
          this.toast.success("Evenement verwijderd");
          this.router.navigate(["organisaties", this.organisation.id, "evenementen"]);
        });
      })
      .catch(() => {});
  }

  newPredefinedSlot() {
    const modalRef = this.modalService.open(FormModalComponent);
    const component = modalRef.componentInstance as FormModalComponent;
    component.title = "Nieuw tijdvak toevoegen";
    component.confirmLabel = "Toevoegen";
    component.fields = [
      new FormModalInputOptions("name", "Naam", "text", "", [Validators.required]),
      new FormModalInputOptions("start", "Starttijd", "datetime", this.event.start, [
        Validators.required,
        DateValidators.isValidFormat,
      ]),
      new FormModalInputOptions("end", "Eindtijd", "datetime", this.event.end, [
        Validators.required,
        DateValidators.isValidFormat,
      ]),
      new FormModalInputOptions(
        "type",
        "Type",
        "select",
        {
          value: "required",
          options: [
            { id: "required", name: "Verplicht" },
            { id: "template", name: "Sjabloon" },
          ].map((type) => new DropdownOption(type.id, type.name)),
        },
        [Validators.required]
      ),
      new FormModalInputOptions("travelMode", "Vervoer", "select", {
        value: this.event.primaryTravelMode,
        options: [
          {
            id: "bicycling",
            name: "Fiets",
          },
          {
            id: "driving",
            name: "Auto",
          },
          {
            id: "walking",
            name: "Te voet",
          },
          {
            id: "transit",
            name: "Openbaar vervoer",
          },
        ].map((type) => new DropdownOption(type.id, type.name)),
      }),
    ];

    component.callback = (result: any) => {
      const slot = {
        id: "",
        name: result.name,
        type: result.type,
        travelMode: result.travelMode,
        start: result.start,
        end: result.end,
      };

      return this.eventsService.addPredefinedSlot(this.organisation.id, this.event.id, slot).pipe(first());
    };
    component.onSuccess = () => {
      this.toast.success("Nieuw tijdvak aangemaakt");
      this.eventsService.getPredefinedSlotsForId(this.organisation.id, this.event.id).subscribe((slots) => {
        this.predefinedSlots = slots;
      });
    };

    component.onError = (err) => this.handleError(err);
  }

  editPredefinedSlot(slot: PredefinedSlot) {
    const modalRef = this.modalService.open(FormModalComponent);
    const component = modalRef.componentInstance as FormModalComponent;
    component.title = "Tijdvak aanpassen";
    component.confirmLabel = "Opslaan";
    component.fields = [
      new FormModalInputOptions("name", "Naam", "text", slot.name, [Validators.required]),
      new FormModalInputOptions("start", "Starttijd", "datetime", slot.start, [
        Validators.required,
        DateValidators.isValidFormat,
      ]),
      new FormModalInputOptions("end", "Eindtijd", "datetime", slot.start, [
        Validators.required,
        DateValidators.isValidFormat,
      ]),
      new FormModalInputOptions(
        "type",
        "Type",
        "select",
        {
          value: slot.type,
          options: [
            { id: "required", name: "Verplicht" },
            { id: "template", name: "Sjabloon" },
          ].map((type) => new DropdownOption(type.id, type.name)),
        },
        [Validators.required]
      ),
      new FormModalInputOptions("travelMode", "Vervoer", "select", {
        value: slot.travelMode,
        options: [
          {
            id: "bicycling",
            name: "Fiets",
          },
          {
            id: "driving",
            name: "Auto",
          },
          {
            id: "walking",
            name: "Te voet",
          },
          {
            id: "transit",
            name: "Openbaar vervoer",
          },
        ].map((type) => new DropdownOption(type.id, type.name)),
      }),
    ];

    component.callback = (result: any) => {
      slot = {
        id: slot.id,
        name: result.name,
        type: result.type,
        travelMode: result.travelMode,
        start: result.start,
        end: result.end,
      };

      return this.eventsService.updatePredefinedSlot(this.organisation.id, this.event.id, slot).pipe(first());
    };
    component.onSuccess = () => {
      this.toast.success("Tijdvak aangepast");
      this.eventsService.getPredefinedSlotsForId(this.organisation.id, this.event.id).subscribe((slots) => {
        this.predefinedSlots = slots;
      });
    };

    component.onError = (err) => this.handleError(err);
  }

  deletePredefinedSlot(slot: PredefinedSlot) {
    const modalRef = this.modalService.open(DeleteModalComponent);
    modalRef.componentInstance.name = slot.name;
    modalRef.result
      .then((result) => {
        this.eventsService
          .deletePredefinedSlot(this.organisation.id, this.event.id, slot)
          .pipe(first())
          .subscribe(() => {
            this.toast.success("Tijdvak verwijderd");
            this.eventsService.getPredefinedSlotsForId(this.organisation.id, this.event.id).subscribe((slots) => {
              this.predefinedSlots = slots;
            });
          });
      })
      .catch(() => {});
  }

  handleError(err: any) {
    const error = err.error;
    if (error.errors) {
      error.errors.forEach((singleError: any) => {
        switch (singleError.message) {
          case "startNotBeforeEnd":
            this.toast.error("Er is iets fout gegaan", "De starttijd is later dan de eindtijd");
            break;
          case "notSameDay":
            this.toast.error("Er is iets fout gegaan", "De start- en eindtijd zijn niet op de zelfde dag");
            break;
          case "revealAtNotBeforeStart":
            this.toast.error(
              "Er is iets fout gegaan",
              "De publicatiedatum van de route is niet voor de start van het evenement"
            );
            break;
          case "publishAtNotBeforeStart":
            this.toast.error(
              "Er is iets fout gegaan",
              "De publicatiedatum van het evenement is niet voor de startdatum"
            );
            break;
          case "startOutsideEvent":
            this.toast.error(
              "Er is iets fout gegaan",
              "De starttijd van dit tijdvak is niet na de startdatum van het evenement"
            );
            break;
          case "endOutsideEvent":
            this.toast.error(
              "Er is iets fout gegaan",
              "De eindtijd van dit tijdvak is niet voor de einddatum van het evenement"
            );
            break;
          case "dateOverlap":
            this.toast.error("Er is iets fout gegaan", "De tijden van dit tijdvak overlappen een ander tijdvak");
            break;
          case "invalidDates":
            this.toast.error("Er is iets fout gegaan", "De ingevulde data zijn ongeldig");
            break;
          default:
            this.toast.error("Er is iets fout gegaan", "Over de fout is niets bekend");
        }
      });
    }
  }

  updateEventInfo() {
    markFormGroupTouched(this.forms.info);

    const { name, start, end, publishAt, revealAt, primaryTravelMode } = this.forms.info.value;

    if (this.forms.info.valid) {
      const changes = {};

      if (this.forms.info.get("name").dirty) {
        changes["name"] = name;
      }

      if (this.forms.info.get("start").dirty) {
        changes["start"] = moment(start, "YYYY-MM-DDTHH:mm").toISOString();
      }

      if (this.forms.info.get("end").dirty) {
        changes["end"] = moment(end, "YYYY-MM-DDTHH:mm").toISOString();
      }

      if (this.forms.info.get("publishAt").dirty) {
        if (publishAt === "" || publishAt === null) {
          changes["publishAt"] = null;
        } else {
          changes["publishAt"] = moment(publishAt, "YYYY-MM-DDTHH:mm").toISOString();
        }
      }

      if (this.forms.info.get("revealAt").dirty) {
        if (revealAt === "" || revealAt === null) {
          changes["revealAt"] = null;
        } else {
          changes["revealAt"] = moment(revealAt, "YYYY-MM-DDTHH:mm").toISOString();
        }
      }

      if (this.forms.info.get("primaryTravelMode").dirty) {
        changes["primaryTravelMode"] = primaryTravelMode;
      }

      this.eventsService
        .updateInOrganisation(this.organisation.id, {
          ...changes,
          id: this.event.id,
        })
        .pipe(take(1))
        .subscribe((event) => {
          this.toast.success("Aanpassing succesvol opgeslagen");
          this.event = event;
          this.resetForm(event);
        }, this.handleError.bind(this));
    }
  }
}
