import { EventEmitter, Injectable } from "@angular/core";
import { HttpClient, HttpHeaders } from "@angular/common/http";

import { Observable } from "rxjs";

import { Event } from "@app/shared/models/event.model";
import { tap } from "rxjs/internal/operators";
import { Slot } from "@app/shared/models/slot.model";
import { PredefinedSlot } from "@app/shared/models/predefined-slot.model";
import { Company } from "../shared/models/company.model";
import { Group } from "../shared/models/group.model";
import { Warning } from "@app/shared/models/warning.model";
import { Member } from "@app/shared/models/member.model";

export interface WarningsResponse {
  counts: { [code: string]: number };
  warnings: { group: Group; warnings: Warning[] }[];
}

@Injectable()
export class EventsDataService {
  companiesWarnings: { [companyId: string]: Warning[] } = {};
  groupsWarnings: { [groupId: string]: Warning[] } = {};
  codesWarnings: { [warningCode: string]: Warning[] } = {};

  _currentEvent: Event = null;
  eventChanged = new EventEmitter();

  get currentEvent(): Event {
    return this._currentEvent;
  }

  set currentEvent(value: Event) {
    this._currentEvent = value;
    this.eventChanged.emit(value);
  }
  constructor(private http: HttpClient) {}

  getAllForOrganisation(organisationId: string, archived: boolean = false): Observable<Event[]> {
    return this.http.get<Event[]>(`/organisations/${organisationId}/events?archived=${archived}`);
  }

  getByIdForOrganisation(organisationId: string, key: string | number): Observable<Event> {
    return this.http.get<Event>(`/organisations/${organisationId}/events/${key}`);
  }

  getAllSlots(organisationId: string, eventId: string): Observable<Group[]> {
    return this.http.get<Group[]>(`/organisations/${organisationId}/events/${eventId}/routes`);
  }

  getWarnings(organisationId: string, eventId: string): Observable<WarningsResponse> {
    return this.http.get<WarningsResponse>(`/organisations/${organisationId}/events/${eventId}/routes/warnings`).pipe(
      tap((w: WarningsResponse) => {
        this.groupsWarnings = {};
        this.companiesWarnings = {};
        this.codesWarnings = {};
        w.warnings.forEach(wg => {
          wg.warnings.forEach(warning => {
            if (!this.groupsWarnings.hasOwnProperty(wg.group.id)) {
              this.groupsWarnings[wg.group.id] = [];
            }
            if (!this.codesWarnings.hasOwnProperty(warning.code)) {
              this.codesWarnings[warning.code] = [];
            }
            warning.groupId = wg.group.id;
            if (warning.code === "overCapacity") {
              // has company
              if (!this.companiesWarnings.hasOwnProperty(warning.data.company.id)) {
                this.companiesWarnings[warning.data.company.id] = [];
              }
              warning.companyId = warning.data.company.id;
              this.companiesWarnings[warning.data.company.id].push(warning);
            }
            this.groupsWarnings[wg.group.id].push(warning);
            this.codesWarnings[warning.code].push(warning);
          });
        });
        return w;
      })
    );
  }

  getAllSlotsByGroupId(organisationId: string, eventId: string, groupId: string): Observable<Group> {
    return this.http.get<Group>(`/organisations/${organisationId}/events/${eventId}/routes/${groupId}`);
  }

  addToOrganisation(organisationId: string, entity: Event): Observable<Event> {
    return this.http.post<Event>(`/organisations/${organisationId}/events`, entity);
  }

  deleteFromOrganisation(organisationId: string, entityId: string): Observable<number> {
    return this.http.delete<number>(`/organisations/${organisationId}/events/${entityId}`);
  }

  getPredefinedSlotsForId(organisationId: string, eventId: string): Observable<PredefinedSlot[]> {
    return this.http.get<PredefinedSlot[]>(`/organisations/${organisationId}/events/${eventId}/predefined-slots`);
  }

  addPredefinedSlot(organisationId: string, eventId: string, slot: PredefinedSlot): Observable<PredefinedSlot> {
    return this.http.post<PredefinedSlot>(`/organisations/${organisationId}/events/${eventId}/predefined-slots`, slot);
  }

  deletePredefinedSlot(organisationId: string, eventId: string, slot: PredefinedSlot): Observable<any> {
    return this.http.delete(`/organisations/${organisationId}/events/${eventId}/predefined-slots/${slot.id}`);
  }

  updatePredefinedSlot(organisationId: string, eventId: string, slot: PredefinedSlot): Observable<PredefinedSlot> {
    return this.http.put<PredefinedSlot>(
      `/organisations/${organisationId}/events/${eventId}/predefined-slots/${slot.id}`,
      slot
    );
  }

  addSlot(organisationId: string, eventId: string, slot: Slot | any): Observable<Slot> {
    return this.http.post<Slot>(`/organisations/${organisationId}/events/${eventId}/slots`, slot);
  }

  generateSlots(organisationId: string, eventId: string): Observable<Slot[]> {
    return this.http.post<Slot[]>(`/organisations/${organisationId}/events/${eventId}/routes/apply-predefined`, {});
  }

  updateSlot(organisationId: string, eventId: string, slot: Slot): Observable<Slot> {
    return this.http.put<Slot>(`/organisations/${organisationId}/events/${eventId}/slots/${slot.id}`, slot);
  }

  deleteSlot(organisationId: string, eventId: string, slotId: string): Observable<void> {
    return this.http.delete<void>(`/organisations/${organisationId}/events/${eventId}/slots/${slotId}`);
  }

  updateInOrganisation(organisationId: string, entity: Partial<Event>) {
    return this.http.put<Event>(`/organisations/${organisationId}/events/${entity.id}`, entity);
  }

  getCompaniesForId(organisationId: string, eventId: string, type: string = "companies"): Observable<Company[]> {
    return this.http.get<Company[]>(`/organisations/${organisationId}/events/${eventId}/${type}`);
  }

  mailCompanies(organisationId: string, eventId: string, companyId?: string): Observable<Company[]> {
    const lastPartOfUrl = companyId ? `companies/${companyId}/send-mail` : "companies/send-mail";
    return this.http.post<Company[]>(`/organisations/${organisationId}/events/${eventId}/${lastPartOfUrl}`, {});
  }

  addCompaniesForId(
    organisationId: string,
    eventId: string,
    companyIds: string[],
    type: string = "companies"
  ): Observable<void> {
    return this.http.post<void>(`/organisations/${organisationId}/events/${eventId}/${type}`, {
      companyIds
    });
  }

  removeCompaniesForId(
    organisationId: string,
    eventId: string,
    companyIds: string[],
    type: string = "companies"
  ): Observable<void> {
    return this.http.request<void>("delete", `/organisations/${organisationId}/events/${eventId}/${type}`, {
      headers: new HttpHeaders({ "Content-Type": "application/json" }),
      body: {
        companyIds
      }
    });
  }

  getUnassignedMembers(organisationId: string, eventId: string, schoolId: string = ""): Observable<Member[]> {
    return this.http.get<Member[]>(
      `/organisations/${organisationId}/events/${eventId}/` +
        `members?accessLevels=["guides","students"]&unassigned=true&schoolId=${schoolId}`
    );
  }

  addMember(organisationId: string, eventId: string, member: Member, schoolId: string = "") {
    return this.http.post<Member>(`/organisations/${organisationId}/events/${eventId}/members`, {
      schoolId,
      userId: member.id
    });
  }

  deleteMember(organisationId: string, eventId: string, memberId: string, schoolId: string = "") {
    return this.http.delete<void>(
      `/organisations/${organisationId}/events/${eventId}/members/${memberId}?schoolId=${schoolId}`
    );
  }
}
