import { Component, OnInit } from "@angular/core";
import { NgbModal } from "@ng-bootstrap/ng-bootstrap";
import { OrganisationsDataService } from "app/organisations/organisations-data.service";
import { Router } from "@angular/router";
import { NotificationsService } from "angular2-notifications";
import { FormControl, FormGroup, Validators } from "@angular/forms";
import * as Color from "color";
import { Event } from "app/shared/models/event.model";
import { Organisation } from "app/shared/models/organisation.model";
import { FormModalComponent, FormModalInputOptions } from "app/shared/form-modal/form-modal.component";
import { EventsDataService } from "app/events/events-data.service";
import { CrudComponent } from "app/shared/crud.component";
import * as moment from "moment";
import { Title } from "@angular/platform-browser";

@Component({
  selector: "app-events",
  templateUrl: "./events.component.html",
  styleUrls: ["./events.component.scss"]
})
export class EventsComponent extends CrudComponent<Event> implements OnInit {
  color = Color("white");
  newEventForm: FormGroup;
  organisation: Organisation;
  archivedEvents: Event[] = [];
  showArchivedEvents = false;

  constructor(
    private organisationsService: OrganisationsDataService,
    private eventsService: EventsDataService,
    private router: Router,
    private title: Title,
    private modal: NgbModal,
    private toast: NotificationsService
  ) {
    super();
    super.initialize(modal);
  }

  ngOnInit() {
    this.newEventForm = new FormGroup({
      name: new FormControl("", Validators.required)
    });

    this.organisation = this.organisationsService.currentOrganisation;
    this.loadEvents();

    this.color = this.organisation.primaryColor;
    this.title.setTitle(`Organisaties > ${this.organisation.name} > Evenementen`);
  }

  loadEvents() {
    this.eventsService.getAllForOrganisation(this.organisation.id).subscribe(events => (this.entities = events));
    this.eventsService
      .getAllForOrganisation(this.organisation.id, true)
      .subscribe(events => (this.archivedEvents = events));
  }

  eventClicked(event: Event) {
    this.router.navigate(["/organisaties", this.organisation.id, "evenementen", event.id]);
  }

  newPressed() {
    const modalRef = this.modal.open(FormModalComponent);
    const component = modalRef.componentInstance as FormModalComponent;
    component.title = "Nieuw evenement toevoegen";
    component.confirmLabel = "Toevoegen";
    component.fields = [
      new FormModalInputOptions("name", "Evenementnaam", "text", "", [Validators.required]),
      new FormModalInputOptions("start", "Begindatum", "date", "", [Validators.required]),
      new FormModalInputOptions("end", "Einddatum", "date", "", [Validators.required])
    ];
    component.callback = (result: any) => {
      const event = new Event();
      event.primaryTravelMode = "bicycling";
      event.name = result.name;
      event.archived = false;
      event.start = moment(result.start)
        .startOf("day")
        .toDate();
      event.end = moment(result.end)
        .endOf("day")
        .toDate();
      return this.eventsService.addToOrganisation(this.organisation.id, event);
    };
    component.onSuccess = (result: any) => {
      this.toast.success(`Evenement '${result.name}' succesvol toegevoegd.`);
      this.loadEvents();
    };
  }

  toggleArchiveEvent(event: Event) {
    this.eventsService
      .updateInOrganisation(this.organisation.id, { id: event.id, archived: !event.archived })
      .subscribe(() => this.loadEvents());
  }
}
