import { Component, OnInit } from "@angular/core";
import { Group } from "app/shared/models/group.model";
import { Slot, slotColors } from "app/shared/models/slot.model";
import { SchoolsDataService } from "app/schools/schools-data.service";
import { tap, flatMap, switchMap, map, first } from "rxjs/operators";
import { of, forkJoin } from "rxjs";
import { School } from "app/shared/models/school.model";
import { EventsDataService } from "../events-data.service";
import { Event, travelModeOptions } from "app/shared/models/event.model";
import * as moment from "moment";
import { NgbModal } from "@ng-bootstrap/ng-bootstrap";
import { FormModalComponent, FormModalInputOptions } from "app/shared/form-modal/form-modal.component";
import { Validators } from "@angular/forms";
import { DateValidators } from "app/shared/validators/date.validator";
import { DropdownOption } from "app/shared/forms/form-input/form-input.component";
import { NotificationsService } from "angular2-notifications";
import { Company } from "app/shared/models/company.model";
import { OrganisationsDataService } from "app/organisations/organisations-data.service";
import { Location } from "app/shared/models/location.model";
import { DeleteModalComponent } from "@app/shared/delete-modal/modal.component";

import { NgxSpinnerService } from "ngx-spinner";
import { Warning } from "@app/shared/models/warning.model";
import { WarningsModalComponent } from "@app/events/planning/warnings-modal/warnings-modal.component";
import { Title } from "@angular/platform-browser";
import { RouteModalComponent } from "@app/shared/route-modal/route-modal.component";
import { Organisation } from "@app/shared/models/organisation.model";
import * as momentTimezone from "moment-timezone";

import { CalendarOptions } from "@fullcalendar/angular";
import { Category } from "../../shared/models/category.model";
import { ResourceSourceInput } from "@fullcalendar/resource-common";

const flatten = (list: any) => list.reduce((a: any, b: any) => a.concat(Array.isArray(b) ? flatten(b) : b), []);

class CalendarEvent {
  title: string;
  start: Date;
  end: Date;
  slot: Slot;
  group: Group;
  id: string;
  isPredefined: boolean;
  index: number;
}

@Component({
  selector: "app-organisation-event-planning",
  templateUrl: "./event-detail-planning.component.html",
  styleUrls: ["./event-detail-planning.component.scss"],
})
export class EventDetailPlanningComponent implements OnInit {
  groups: Group[];
  filteredGroups: Group[];
  warnings: { group: Group; warnings: Warning[] }[] = [];
  warningCount: { [code: string]: number };
  event: Event;
  organisation: Organisation;
  viewFromDestinations = false;
  slots: CalendarEvent[] = [];
  slotColors = slotColors;

  schools: School[] = [];
  locations: Location[] = [];
  companies: Company[] = [];
  destinations: { title: string; options: DropdownOption[] }[] = [];

  usedDestinations: (School | Location | Company)[] = [];
  filteredDestinations: (School | Location | Company)[] = [];

  queryString = "";

  calendarOptions: CalendarOptions = {
    schedulerLicenseKey: "CC-Attribution-NonCommercial-NoDerivatives",
    initialView: "resourceTimelineDay",
    height: "100%",
    locale: "nl",
    slotLabelInterval: "00:30",
    resourceOrder: ["title"],
    slotLabelFormat: {
      hour: "numeric",
      minute: "2-digit",
      meridiem: false,
    },
    eventTimeFormat: {
      hour: "numeric",
      minute: "2-digit",
      meridiem: false,
    },
    headerToolbar: {
      left: "title",
      center: "",
      right: "",
    },
    resourceAreaWidth: 500,
    resourceLaneClassNames: (arg) => (arg.resource.id === "_predefinedSlots" ? "grayed-out" : ""),
    resourceAreaColumns: [
      {
        headerContent: "Groepen",
        field: "title",
        cellClassNames: (arg) => {
          if (arg.resource.id === "_predefinedSlots") {
            return "grayed-out";
          }
          if (this.groups.map((g) => g.id).includes(arg.resource.id)) {
            return "clickable";
          }
          return "";
        },
        cellDidMount: (arg) => {
          arg.el.addEventListener("click", (event) => {
            const resourceId = arg.resource.id;
            const group = this.groups.find((g) => g.id === resourceId);
            if (group) {
              this.editRouteForGroup(group);
            }
          });
        },
      },
      {
        headerContent: "Categorieën",
        width: 150,
        cellClassNames: (arg) => (arg.resource.id === "_predefinedSlots" ? "grayed-out" : ""),
        cellContent: (arg) => {
          const group =
            this.groups.find((g) => arg.resource.id === g.id) ||
            this.companies.find((g) => arg.resource.id === g.id) ||
            null;

          if (!group || group.categories.length === 0) {
            return "";
          }
          return {
            html:
              "<div><div class='category-circles' style='padding-top: 6px;'>" +
              group.categories
                .map(
                  (category: Category) =>
                    "<div " +
                    'style="background-color:' +
                    category.color +
                    ";display:inline-block; margin-left: 8px; height: 12px;" +
                    ' border-radius: 15px; width: 12px;" data-toggle="tooltip"' +
                    ' title="' +
                    category.name +
                    '"></div>'
                )
                .join("") +
              "</div></div>",
          };
        },
      },
      {
        headerContent: "",
        width: 40,
        cellClassNames: (arg) => (arg.resource.id === "_predefinedSlots" ? "grayed-out" : ""),
        cellContent: (arg) => {
          const isGroup = this.groups.findIndex((g) => g.id === arg.resource.id) !== -1;
          if (!isGroup) {
            return "";
          }

          return {
            html: `<button data-id="${arg.resource.id}" title="Tijdvak toevoegen" style="border: 0; margin-top: 1px;" class="btn btn-link add-slot-btn"><i data-id="${arg.resource.id}" class="fa fa-plus"></i></button>`,
          };
        },
        cellDidMount: (arg) => {
          arg.el.querySelector("button.add-slot-btn")?.addEventListener("click", (event) => {
            const groupId = (event.target as HTMLElement).getAttribute("data-id");
            if (groupId) {
              this.newSlotClicked(this.groups.find((g) => g.id === groupId));
            }
          });
        },
      },
      {
        headerContent: "",
        width: 30,
        cellClassNames: (arg) => (arg.resource.id === "_predefinedSlots" ? "grayed-out" : ""),
        cellContent: (arg) => {
          if (!this.warnings || arg.resource.id === "_predefinedSlots") {
            return "";
          }
          const isCompany = this.companies.findIndex((g) => g.id === arg.resource.id) !== -1;
          const warnings = isCompany
            ? this.eventsService.companiesWarnings[arg.resource.id]
            : this.eventsService.groupsWarnings[arg.resource.id];

          if (!warnings || warnings.length === 0) {
            return "";
          }
          let color = "gray";
          const warningCodes = warnings.map((w) => w.code);
          if (
            warningCodes.includes("outsideEvent") ||
            warningCodes.includes("overCapacity") ||
            warningCodes.includes("required")
          ) {
            color = "darkred";
          } else {
            color = "#c09500";
          }

          return {
            html: `<div title="Er zijn waarschuwingen voor deze route." data-id="${arg.resource.id}" data-type="${
              isCompany ? "company" : "group"
            }" class="warnings-button" style="color: ${color};cursor: pointer;text-align: center;"><i data-id="${
              arg.resource.id
            }" data-type="${
              isCompany ? "company" : "group"
            }" style="font-size: 75%; position: relative; top:5px;" class="fa fa-exclamation-triangle"></i></div>`,
          };
        },
        cellDidMount: (arg) => {
          arg.el.querySelector("div.warnings-button")?.addEventListener("click", (event) => {
            const entityId = (event.target as HTMLDivElement).getAttribute("data-id");
            const type = (event.target as HTMLDivElement).getAttribute("data-type");

            if (!type || !entityId) {
              return;
            }

            const modalRef = this.modalService.open(WarningsModalComponent, { size: "lg" });
            const component = modalRef.componentInstance as WarningsModalComponent;

            if (type === "group") {
              component.warnings = flatten(Object.values(this.eventsService.groupsWarnings[entityId]));
            } else {
              component.warnings = flatten(Object.values(this.eventsService.companiesWarnings[entityId]));
            }

            component.groups = this.groups;
            component.companies = this.companies;
          });
        },
      },
    ],
    eventClick: (arg) => {
      this.eventClicked(arg.event._def.extendedProps.slot);
    },
  };

  constructor(
    private schoolsService: SchoolsDataService,
    public eventsService: EventsDataService,
    private modalService: NgbModal,
    private spinner: NgxSpinnerService,
    private toast: NotificationsService,
    private title: Title,
    private organisationsService: OrganisationsDataService
  ) {}

  ngOnInit() {
    this.organisation = this.organisationsService.currentOrganisation;

    this.groups = [];

    this.initialize(this.eventsService.currentEvent);
    this.spinner.show();
  }

  toggleView() {
    this.viewFromDestinations = !this.viewFromDestinations;
    this.initCalendar();
  }

  initialize(event: Event) {
    this.title.setTitle(`Organisaties > ${this.organisation.name} > Evenementen > ${event.name} > Planning`);

    this.event = event;
    this.calendarOptions.initialDate = moment(this.event.start).format("YYYY-MM-DD");

    // Load all companies and locations
    forkJoin([
      this.organisationsService.getLocations(this.organisation.id),
      this.eventsService.getCompaniesForId(this.organisation.id, this.event.id),
    ]).subscribe(([locations, companies]) => {
      this.locations = locations as Location[];
      this.companies = companies as Company[];
    });

    this.schoolsService
      .getAllForOrganisation(this.organisation.id)
      .pipe(
        tap((schools) => (this.schools = schools)),
        flatMap((school) => school),
        flatMap((school: School) => {
          return of(school).pipe(
            switchMap((s) => this.schoolsService.getAllGroupsWithId(this.organisation.id, s.id, this.event.id)),
            map((groups) => {
              return groups.map((group) => {
                group.school = school;
                return group;
              });
            })
          );
        }),
        flatMap((group) => group)
      )
      .subscribe(
        (group: Group) => this.groups.push(group),
        () => {},
        () => {
          this.groups.sort((a, b) => a.name.localeCompare(b.name));
          this.filteredGroups = this.groups;
          this.initCalendar();
          setTimeout(() => this.getEvents(), 100);
        }
      );
    this.eventsService.getWarnings(this.organisation.id, this.event.id).subscribe((response) => {
      this.warnings = response.warnings;
      this.warningCount = response.counts;
    });
  }

  totalWarningCount() {
    let count = 0;
    if (this.warningCount) {
      Object.values(this.warningCount).forEach((c) => (count += c));
    }
    return count;
  }

  showWarningsForEvent() {
    const modalRef = this.modalService.open(WarningsModalComponent, { size: "lg" });
    const component = modalRef.componentInstance as WarningsModalComponent;
    component.warnings = flatten(Object.values(this.eventsService.codesWarnings));

    component.groups = this.groups;
    component.companies = this.companies;
    component.counts = this.warningCount;
  }

  initDestinations() {
    this.destinations = [];
    this.destinations.push({
      title: "Scholen",
      options: this.schools.map((entity) => new DropdownOption(entity.id, entity.name)),
    });
    this.destinations.push({
      title: "Bedrijven",
      options: this.companies.map((entity) => new DropdownOption(entity.id, entity.name)),
    });
    this.destinations.push({
      title: "Locaties",
      options: this.locations.map((entity) => new DropdownOption(entity.id, entity.name)),
    });
  }

  getEvents() {
    forkJoin([
      this.eventsService.getPredefinedSlotsForId(this.organisation.id, this.event.id).pipe(
        map((slots) => {
          return slots.map((slot) => {
            const instance = new Slot(slot as Slot);
            return {
              title: slot.name,
              start: moment(slot.start).toDate(),
              end: moment(slot.end).toDate(),
              slot: instance,
              group: { id: "_predefinedSlots" },
              id: slot.id,
              isPredefined: true,
            };
          });
        })
      ),
      this.eventsService
        .getAllSlots(this.organisation.id, this.event.id)
        .pipe(map((groups) => this.getSlotsFromGroups(groups))),
    ])
      .pipe(
        map((results: any[]) => {
          return flatten(results);
        })
      )
      .subscribe((slots: CalendarEvent[]) => {
        this.slots = slots;
        this.initDestinations();
        this.spinner.hide();
        this.initCalendar();

        this.eventsService.getWarnings(this.organisation.id, this.event.id).subscribe((response) => {
          this.warnings = response.warnings;
          this.warningCount = response.counts;
        });
      });
  }

  getSlotsFromGroups(g: Group[]) {
    return g.map((group) => {
      if (!group.slots) {
        group.slots = [];
      }
      const slots = group.slots.map((slot) => {
        const instance = new Slot(slot);
        return {
          title: slot.name,
          start: moment(slot.start).toDate(),
          end: moment(slot.end).toDate(),
          slot: instance,
          group: group,
          id: slot.id,
          isPredefined: false,
          index: this.filteredGroups.findIndex((gi) => gi.id === group.id),
        };
      });
      return slots;
    });
  }

  getAllSlotsForGroup(g: Group) {
    return this.eventsService.getAllSlotsByGroupId(this.organisation.id, this.event.id, g.id).pipe(
      map((group) => {
        if (!group.slots) {
          group.slots = [];
        }
        const slots = group.slots.map((slot) => {
          const instance = new Slot(slot);
          return {
            title: slot.name,
            start: moment(slot.start).toDate(),
            end: moment(slot.end).toDate(),
            slot: instance,
            group: group,
            id: slot.id,
            isPredefined: false,
            index: this.filteredGroups.findIndex((gi) => gi.id === group.id),
          };
        });
        return slots;
      })
    );
  }

  reloadGroupSlots(g: Group) {
    // Clear old records
    this.removeAllSlotsForGroupLocally(g);

    // Add new records
    this.getAllSlotsForGroup(g).subscribe(
      (slots) => (this.slots = this.slots.concat(slots)),
      () => {},
      () => {
        this.eventsService.getWarnings(this.organisation.id, this.event.id).subscribe((response) => {
          this.warnings = response.warnings;
          this.warningCount = response.counts;
          this.initCalendar();
        });
      }
    );
  }

  removeAllSlotsForGroupLocally(g: Group) {
    this.slots = this.slots.filter((e) => !e.group || e.group.id !== g.id);
  }

  initCalendar() {
    if (this.viewFromDestinations) {
      this.usedDestinations = this.slots.map((s) => s.slot.location || s.slot.school || s.slot.company);
      this.usedDestinations = this.usedDestinations.filter((elem, pos, arr) => arr.indexOf(elem) === pos);
      if (!this.usedDestinations[0]) {
        this.usedDestinations = this.usedDestinations.slice(1);
      }
      this.filteredDestinations = this.usedDestinations.filter(
        (d) => d.name.toLowerCase().indexOf(this.queryString) !== -1 || !this.queryString
      );

      this.calendarOptions.resources = this.filteredDestinations.map((destination) => {
        return { id: destination.id, title: destination.name };
      });
      this.calendarOptions.events = this.slots.map((slot) => {
        const content = slot.slot.cardContent;
        const destination = slot.slot.location || slot.slot.school || slot.slot.company;
        if (destination) {
          return {
            resourceId: destination.id,
            color: content.color,
            slot: slot,
            icon: content.icon,
            title: `${slot.group.name} (${this.groups.find((g) => g.id === slot.group.id).memberCount})`,
            start: slot.start,
            end: slot.end,
          };
        }
      });
      this.calendarOptions.events = this.calendarOptions.events.filter(
        (elem: any, pos: any, arr: any) => arr.indexOf(elem) === pos
      );
      if (!this.calendarOptions.events[0]) {
        this.calendarOptions.events = this.calendarOptions.events.slice(1);
      }
      this.calendarOptions.resourceAreaColumns[0].headerContent = "Bestemmingen";

      // this.calendarOptions.resourceRender = null;
    } else {
      const resources: ResourceSourceInput = [
        { id: "_predefinedSlots", title: "Voorgedefinieerde tijdvakken", eventClassNames: "asdf" },
      ];
      this.calendarOptions.resources = resources.concat(
        this.filteredGroups.map((group) => {
          return { id: group.id, title: `${group.name} (${group.memberCount})` };
        })
      );

      this.calendarOptions.events = this.slots.map((slot) => {
        const content = slot.slot.cardContent;
        return {
          resourceId: slot.group.id,
          color: content.color,
          slot: { ...slot },
          icon: content.icon,
          title: content.title,
          start: slot.start,
          end: slot.end,
        };
      });

      this.calendarOptions.resourceAreaColumns[0].headerContent = "Groepen";
    }

    const isSameDay =
      momentTimezone(this.event.start).tz("Europe/Amsterdam").startOf("day") ===
      momentTimezone(this.event.end).tz("Europe/Amsterdam").startOf("day");
    if (isSameDay) {
      this.calendarOptions.validRange = {
        start: moment(this.event.start).startOf("day").toDate(),
        end: moment(this.event.end).endOf("day").toDate(),
      };
    }
  }

  updateFilter(event: any) {
    const val = event.target.value.toLowerCase();
    this.queryString = val;

    // filter our data
    const temp = this.groups.filter((d) => {
      return d.name.toLowerCase().indexOf(val) !== -1 || !val;
    });

    // update the rows
    this.filteredGroups = temp;

    this.eventsService.getWarnings(this.organisation.id, this.event.id).subscribe((response) => {
      this.warnings = response.warnings;
      this.warningCount = response.counts;
      this.initCalendar();
    });
  }

  getMappedDestinationId(destinationId: string) {
    const location = this.locations.find((e) => e.id === destinationId);
    const school = this.schools.find((e) => e.id === destinationId);
    const company = this.companies.find((e) => e.id === destinationId);

    return {
      locationId: location ? location.id : null,
      schoolId: school ? school.id : null,
      companyId: company ? company.id : null,
    };
  }

  getDestinationEntityById(destinationId: string) {
    const company = this.companies.find((e) => e.id === destinationId);
    if (company) {
      return company;
    }
    const school = this.schools.find((e) => e.id === destinationId);
    if (school) {
      return school;
    }
    const location = this.locations.find((e) => e.id === destinationId);
    if (location) {
      return location;
    }

    return null;
  }

  newSlotClicked(group: Group) {
    const modalRef = this.modalService.open(FormModalComponent);
    const component = modalRef.componentInstance as FormModalComponent;
    component.title = "Nieuw tijdvak voor " + group.name;
    component.confirmLabel = "Opslaan";
    component.fields = [
      new FormModalInputOptions("name", "Naam", "text", "", [Validators.required]),
      new FormModalInputOptions("start", "Starttijd", "datetime", this.event.start, [Validators.required]),
      new FormModalInputOptions("end", "Eindtijd", "datetime", this.event.end, [Validators.required]),
      new FormModalInputOptions(
        "travelMode",
        "Vervoer",
        "select",
        {
          value: this.event.primaryTravelMode,
          options: travelModeOptions.map((type) => new DropdownOption(type.value, type.label)),
        },
        [Validators.required]
      ),
      new FormModalInputOptions("destinationId", "Bestemming", "select", {
        value: "",
        optionGroups: this.destinations,
      }),
    ];
    component.callback = (result: any) => {
      const data = {
        id: "",
        name: result.name,
        travelMode: result.travelMode,
        start: moment(result.start, "DD-MM-YYYY HH:mm").toDate(),
        end: moment(result.end, "DD-MM-YYYY HH:mm").toDate(),
        groupId: group.id,
        ...this.getMappedDestinationId(result.destinationId),
      };

      // @ts-ignore
      return this.eventsService.addSlot(this.organisation.id, this.event.id, data).pipe(first());
    };
    component.onSuccess = (newSlot) => {
      this.toast.success("Tijdvak aangemaakt");
      this.reloadGroupSlots(group);
    };
    component.onError = (err: any) => this.handleError(err);
  }

  editRouteForGroup(group: Group): void {
    const modalRef = this.modalService.open(RouteModalComponent);
    const component = modalRef.componentInstance as RouteModalComponent;
    component.destinations = []
      .concat(...this.destinations.map((dg) => dg.options))
      .map((d) => this.getDestinationEntityById(d.value));
    component.title = "Route aanpassen voor " + group.name;
    component.groupUrl = `/organisaties/${this.organisation.id}/scholen/${group.school.id}/groepen/${group.id}`;
    component.confirmLabel = "Opslaan";

    const slotsForGroup = this.slots.filter((s) => s.group.id === group.id);
    component.slots = slotsForGroup.map((s) => s.slot);
    component.fields = [
      ...slotsForGroup.map((s) => {
        const currentDestination = s.slot.company || s.slot.location || s.slot.school;
        return new FormModalInputOptions(s.slot.id, s.slot.name || "", "select", {
          value: currentDestination ? currentDestination.id : "",
          optionGroups: this.destinations,
        });
      }),
    ];
    component.callback = (result: any) => {
      const updates = Object.keys(result).map((slotId) => {
        const data = {
          id: slotId,
          ...this.getMappedDestinationId(result[slotId]),
        };

        // @ts-ignore
        return this.eventsService.updateSlot(this.organisation.id, this.event.id, data).pipe(first());
      });

      return forkJoin(updates);
    };
    component.onSuccess = () => {
      this.toast.success("Route aangepast");
      this.reloadGroupSlots(group);
    };
    component.onError = (err: any) => this.handleError(err);
  }

  eventClicked(event: any): void {
    const { slot } = event;
    if (event.isPredefined) {
      const modalRef = this.modalService.open(FormModalComponent);
      const component = modalRef.componentInstance as FormModalComponent;
      component.title = "Voorgedefinieerd tijdvak toepassen";
      component.confirmLabel = "Opslaan";
      component.fields = [
        new FormModalInputOptions(
          "groupId",
          "Groep",
          "multiselect",
          {
            value: this.filteredGroups.length > 0 ? [this.filteredGroups[0].id] : "",
            options: this.groups.map((type) => new DropdownOption(type.id, type.name)),
          },
          [Validators.required]
        ),
        new FormModalInputOptions("name", "Naam", "text", slot.name || "", null, { disabled: true }),
        new FormModalInputOptions("start", "Starttijd", "datetime", slot.start, null, { disabled: true }),
        new FormModalInputOptions("end", "Eindtijd", "datetime", slot.end, null, { disabled: true }),
        new FormModalInputOptions(
          "travelMode",
          "Vervoer",
          "select",
          {
            value: slot.travelMode || this.event.primaryTravelMode,
            options: travelModeOptions.map((type) => new DropdownOption(type.value, type.label)),
          },
          [Validators.required]
        ),
      ];
      component.callback = (result: any) => {
        const results = result.groupId.map((gId: any) => {
          const data = {
            id: "",
            name: slot.name,
            travelMode: result.travelMode,
            start: slot.start,
            end: slot.end,
            groupId: gId,
          };

          return this.eventsService.addSlot(this.organisation.id, this.event.id, data).pipe(first());
        });

        // @ts-ignore
        return forkJoin(results);
      };
      component.onSuccess = (newSlot) => {
        this.toast.success("Voorgedefineerd tijdvak toegepast");
        this.getEvents();
      };
      component.onError = (err: any) => this.handleError(err);
    } else {
      const modalRef = this.modalService.open(FormModalComponent);
      const component = modalRef.componentInstance as FormModalComponent;
      component.title = "Tijdvak aanpassen";
      component.confirmLabel = "Opslaan";

      const currentDestination = slot.company || slot.location || slot.school;

      component.fields = [
        new FormModalInputOptions("name", "Naam", "text", slot.name || ""),
        new FormModalInputOptions("start", "Starttijd", "datetime", slot.start, [
          Validators.required,
          DateValidators.isValidFormat,
        ]),
        new FormModalInputOptions("end", "Eindtijd", "datetime", slot.end, [
          Validators.required,
          DateValidators.isValidFormat,
        ]),
        new FormModalInputOptions("travelMode", "Vervoer", "select", {
          value: slot.travelMode || "",
          options: travelModeOptions.map((type) => new DropdownOption(type.value, type.label)),
        }),
        new FormModalInputOptions("destinationId", "Bestemming", "select", {
          value: currentDestination ? currentDestination.id : "",
          optionGroups: this.destinations,
        }),
      ];
      component.callback = (result: any) => {
        const data = {
          id: slot.id,
          name: result.name,
          start: result.start,
          end: result.end,
          travelMode: result.travelMode,
          ...this.getMappedDestinationId(result.destinationId),
        };

        // @ts-ignore
        return this.eventsService.updateSlot(this.organisation.id, this.event.id, data).pipe(first());
      };
      component.onSuccess = (editedSlot) => {
        this.toast.success("Tijdvak aangepast");
        this.reloadGroupSlots(event.group);
      };
      component.onError = (err: any) => this.handleError(err);
      component.hasDeleteCallback = true;
      component.deleteCallback = () => {
        this.deleteSlot(slot);
        return of();
      };
    }
  }

  deleteUnusedSlots() {
    const modalRef = this.modalService.open(FormModalComponent);
    const component = modalRef.componentInstance as FormModalComponent;
    component.title = "Tijdvakken verwijderen";
    component.description = "Alle tijdvakken zonder bestemming worden hiermee verwijderd.";
    component.confirmLabel = "Verwijderen";

    const slotsForGroup = this.slots.filter(
      (s) => !(s.slot.company || s.slot.location || s.slot.school) && !s.isPredefined
    );
    component.callback = (result: any) => {
      return forkJoin(
        slotsForGroup.map((s) => this.eventsService.deleteSlot(this.organisation.id, this.event.id, s.id))
      );
    };
    component.onSuccess = () => {
      this.toast.success("Tijdvakken verwijderd");
      this.getEvents();
    };
    component.onError = (err: any) => this.handleError(err);
  }

  deleteSlot(slot: Slot) {
    const modalRef = this.modalService.open(DeleteModalComponent);
    const component = modalRef.componentInstance as DeleteModalComponent;
    component.name = `Tijdvak van ${moment(slot.start).format("HH:mm")} tot ${moment(slot.end).format("HH:mm")}`;
    modalRef.result
      .then((result) => {
        this.eventsService.deleteSlot(this.organisation.id, this.event.id, slot.id).subscribe(() => {
          this.toast.success(`Tijdvak is succesvol verwijderd.`);
          this.modalService.dismissAll();
          this.getEvents();
        }, this.handleError.bind(this));
      })
      .catch(() => {});
  }

  handleError(err: any) {
    const error = err.error;
    if (error.errors) {
      error.errors.forEach((singleError: any) => {
        switch (singleError.message) {
          case "requiredSlotOverlap":
            this.toast.error("Er is iets fout gegaan", "Deze tijden overlappen een verplicht voorgedefineerd tijdvak");
            break;
          case "multipleLocationsSet":
            this.toast.error("Er is iets fout gegaan", "Een tijdvak mag maar 1 locatie toewijzing hebben");
            break;
          case "dateOverlap":
            this.toast.error("Er is iets fout gegaan", "Deze tijden overlappen met een ander tijdvak voor deze groep");
            break;
          case "startOutsideEvent":
            this.toast.error(
              "Er is iets fout gegaan",
              "De starttijd van dit tijdvak is niet na de startdatum van het evenement"
            );
            break;
          case "endOutsideEvent":
            this.toast.error(
              "Er is iets fout gegaan",
              "De eindtijd van dit tijdvak is niet voor de einddatum van het evenement"
            );
            break;
          case "invalidDates":
            this.toast.error("Er is iets fout gegaan", "De ingevulde data zijn ongeldig");
            break;
          default:
            this.toast.error("Er is iets fout gegaan", "Over de fout is niets bekend");
        }
      });
    }
  }

  generateSlots() {
    this.eventsService.generateSlots(this.organisation.id, this.event.id).subscribe(() => {
      this.toast.success("Voorgedefineerde tijdvakken toegepast");
      this.spinner.show();
      this.getEvents();
    });
  }
}
