import { Component, OnInit, Input, OnDestroy } from "@angular/core";
import { EmailBatch } from "@app/shared/emails/email-batch.model";
import { EmailsDataService } from "@app/shared/emails/emails-data.service";
import { OrganisationsDataService } from "@app/organisations/organisations-data.service";
import { ActivatedRoute, Router } from "@angular/router";
import { Organisation } from "@app/shared/models/organisation.model";
import { Subscription } from "rxjs";
import { Email } from "@app/shared/emails/email.model";

@Component({
  selector: "app-email-batch",
  template: `
    <div class="batch-detail-wrapper card">
      <div class="card-header">
        <div class="header-row">
          <div>
            <h5 class="card-title">{{ batch.name }}</h5>
            <div class="status-row">
              <div class="status-item" placement="top" ngbTooltip="Verstuurd op">
                <i class="fa fa-paper-plane"></i> {{ batch.sentAt | dutchDate: true }}
              </div>
              <div class="status-item" placement="top" ngbTooltip="Laatste activiteit">
                <i class="fa fa-history"></i> {{ batch.lastActivity | dutchDate: true }}
              </div>
              <div class="status-item-small" placement="top" ngbTooltip="Totaal verstuurd">
                <i class="fa fa-inbox"></i> {{ batch.totalEmails }}
              </div>
              <div class="status-item-small" placement="top" ngbTooltip="Ongelezen">
                <i class="fa fa-envelope"></i> {{ batch.unreadEmails }}
              </div>
            </div>
          </div>
          <button (click)="closeBatch()" class="btn btn-light"><i class="fa fa-times"></i></button>
        </div>
      </div>
      <table class="table table-fluid table-bordered table-sm status-table">
        <thead>
          <tr>
            <td class="status-table-header" colspan="4" (click)="statusTableVisible = !statusTableVisible">
              Status voor de <strong>{{ batch.initiallySent }}</strong> initiële mails
              <i class="fa fa-chevron-{{ statusTableVisible ? 'up' : 'down' }}"></i>
            </td>
          </tr>
        </thead>
        <tbody *ngIf="statusTableVisible">
          <tr>
            <td
              colspan="4"
              [ngClass]="{ 'semi-transparent': filter && filter !== 'sent' }"
              class="table-info"
              (click)="toggleStatusFilter('sent')"
            >
              <div [ngbTooltip]="statuses.sent.tooltip">{{ statuses.sent.translation }}: {{ statuses.sent.count }}</div>
            </td>
          </tr>
          <tr>
            <td
              class="table-success"
              [ngClass]="{ 'semi-transparent': filter && filter !== 'delivered' }"
              colspan="2"
              (click)="toggleStatusFilter('delivered')"
            >
              <div [ngbTooltip]="statuses.delivered.tooltip">
                {{ statuses.delivered.translation }}: {{ statuses.delivered.count }}
                <i class="fa fa-caret-down table-icon"></i>
                <i class="fa fa-caret-down table-icon-bottom"></i>
              </div>
            </td>
            <td
              class="has-arrow-down table-warning"
              [ngClass]="{ 'semi-transparent': filter && filter !== 'delayed' }"
              (click)="toggleStatusFilter('delayed')"
            >
              <div [ngbTooltip]="statuses.delayed.tooltip">
                {{ statuses.delayed.translation }}: {{ statuses.delayed.count }}
                <i class="fa fa-caret-down table-icon"></i>
              </div>
            </td>
            <td
              class="has-arrow-down table-danger"
              [ngClass]="{ 'semi-transparent': filter && filter !== 'rejected' }"
              (click)="toggleStatusFilter('rejected')"
            >
              <div [ngbTooltip]="statuses.rejected.tooltip">
                {{ statuses.rejected.translation }}: {{ statuses.rejected.count }}
                <i class="fa fa-caret-down table-icon"></i>
              </div>
            </td>
          </tr>
          <tr>
            <td
              class="has-arrow-down table-success"
              colspan="4"
              [ngClass]="{ 'semi-transparent': filter && filter !== 'opened' }"
              (click)="toggleStatusFilter('opened')"
            >
              <div [ngbTooltip]="statuses.opened.tooltip">
                {{ statuses.opened.translation }}: {{ statuses.opened.count }}
              </div>
            </td>
          </tr>
          <tr>
            <td
              class="has-arrow-down table-success"
              colspan="4"
              [ngClass]="{ 'semi-transparent': filter && filter !== 'clicked' }"
              (click)="toggleStatusFilter('clicked')"
            >
              <div [ngbTooltip]="statuses.clicked.tooltip">
                {{ statuses.clicked.translation }}: {{ statuses.clicked.count }}
                <i class="fa fa-caret-down table-icon"></i>
              </div>
            </td>
          </tr>
        </tbody>
      </table>

      <div class="email-list">
        <app-email-thread *ngFor="let mail of filteredEmails" [email]="mail"></app-email-thread>
      </div>
    </div>
  `,
  styleUrls: ["./email-batch.component.scss"]
})
export class EmailBatchComponent implements OnInit, OnDestroy {
  batch: EmailBatch = new EmailBatch();
  filteredEmails: Email[] = [];
  statusTableVisible = false;
  statuses = {
    sent: {
      translation: "Verzonden",
      tooltip: "Vanuit het systeem verzonden, nog geen verdere status bekend",
      count: 0
    },
    delivered: { translation: "Afgeleverd", tooltip: "Afgeleverd bij ontvanger", count: 0 },
    opened: { translation: "Geopend", tooltip: "Geopend door ontvanger", count: 0 },
    clicked: { translation: "Aangeklikt", tooltip: "Ontvanger heeft iets aangeklikt", count: 0 },
    delayed: { translation: "Vertraagd", tooltip: "Mail is vertraagd, nog niet aangekomen", count: 0 },
    rejected: { translation: "Geweigerd", tooltip: "Mail is geweigerd door ontvanger", count: 0 }
  };

  filter: string = null;

  organisation: Organisation;
  paramSubscription: Subscription;

  constructor(
    private emailsService: EmailsDataService,
    private organisationsService: OrganisationsDataService,
    private route: ActivatedRoute,
    private router: Router
  ) {}

  ngOnInit() {
    this.organisation = this.organisationsService.currentOrganisation;

    this.paramSubscription = this.route.params.subscribe(params => {
      const id = params.batchId;
      this.emailsService.getBatchByIdForOrganisation(this.organisation.id, id).subscribe(batchDetail => {
        this.batch = { ...this.emailsService.batches.find(b => b.id === id), ...batchDetail };
        Object.keys(this.batch.statusCounts).forEach(status => {
          this.statuses[status].count = this.batch.statusCounts[status];
        });
        this.filteredEmails = this.batch.emails;
      });
    });
  }

  toggleStatusFilter(status: string) {
    this.filter = this.filter === status ? "" : status;
    console.log(this.filter);
    if (this.filter) {
      this.filteredEmails = this.batch.emails.filter(email => email.status === this.filter);
    } else {
      this.filteredEmails = this.batch.emails;
    }
  }

  ngOnDestroy() {
    this.paramSubscription.unsubscribe();
  }

  closeBatch() {
    this.router.navigate(["/organisaties", this.organisation.id, "communicatie"]);
  }
}
