import { Component, OnInit, Input } from "@angular/core";
import { Email, EmailStatuses, renderEmail } from "@app/shared/emails/email.model";
import * as Markdown from "markdown-it";
import { EmailsDataService } from "@app/shared/emails/emails-data.service";
import { OrganisationsDataService } from "@app/organisations/organisations-data.service";
import { FormGroup, FormControl, Validators } from "@angular/forms";
import { markFormGroupTouched } from "@app/shared/form-group";
import { DropdownOption } from "@app/shared/forms/form-input/form-input.component";
import { NotificationsService } from "angular2-notifications";

@Component({
  selector: "app-email-thread",
  templateUrl: "./email-thread.component.html",
  styleUrls: ["./email-thread.component.scss"]
})
export class EmailThreadComponent implements OnInit {
  @Input() email: Email = null;
  expanded = false;
  md: Markdown;
  mails: Email[] = [];
  replyExpanded = false;
  replyForm: FormGroup;
  recipientNames: { [key: string]: string } = {};
  recipients: DropdownOption[] = [];
  organisationId: string;

  constructor(
    private organisationsService: OrganisationsDataService,
    private emailsService: EmailsDataService,
    private toast: NotificationsService
  ) {
    this.md = new Markdown({ html: false });
  }

  ngOnInit() {
    this.organisationId = this.organisationsService.currentOrganisation.id;
  }

  getMarkdownParsedValue(body: string) {
    return this.md.render(body);
  }

  getEmailStatus(status: string) {
    return EmailStatuses[status].translation;
  }

  getEmailPreview() {
    const body = this.email.body.replace("\n", " ").slice(0, 100);
    const md = this.md.render(`${body} ...`);
    return md;
  }

  getRenderedEmail() {
    const md = this.md.render(this.email.body);
    // return renderEmail(md);
    return md;
  }

  toggleExpand() {
    this.expanded = !this.expanded;
    if (this.expanded) {
      this.emailsService
        .getThreadByIdForOrganisation(
          this.organisationsService.currentOrganisation.id,
          this.email.batchId,
          this.email.id
        )
        .subscribe(mailThread => {
          this.email = { ...this.email, ...mailThread[0] };
          this.mails = mailThread.slice(1);
          if (this.mails.length > 0) {
            const recipientsMap = {};
            this.recipientNames = {};
            this.recipients = [];

            this.mails.forEach(m => {
              recipientsMap[m.senderEmail] = new DropdownOption(m.senderEmail, `${m.senderName} <${m.senderEmail}>`);
              recipientsMap[m.recipientEmail] = new DropdownOption(
                m.recipientEmail,
                `${m.recipientName} <${m.recipientEmail}>`
              );
              this.recipientNames[m.senderEmail] = m.senderName;
              this.recipientNames[m.recipientEmail] = m.recipientName;
            });

            this.recipients = Object.values(recipientsMap);
            this.replyForm = new FormGroup({
              recipientEmail: new FormControl(this.recipients[0].value, Validators.required),
              subject: new FormControl(`Re: ${this.mails[this.mails.length - 1].subject}`, Validators.required),
              body: new FormControl("", Validators.required)
            });
          }
        });
    }
  }

  toggleRead(mail: Email) {
    this.emailsService
      .markEmailAsRead(
        this.organisationsService.currentOrganisation.id,
        this.email.batchId,
        this.email.id,
        mail.id,
        !mail.read
      )
      .subscribe(() => {
        mail.read = !mail.read;
      });
  }

  translateEmailStatus(status: string) {
    return EmailStatuses[status].translation;
  }

  submitReply() {
    markFormGroupTouched(this.replyForm);

    if (this.replyForm.valid) {
      const values = this.replyForm.value;

      const data = {
        emailId: this.mails[this.mails.length - 1].id,
        recipientEmail: values.recipientEmail,
        recipientName: this.recipientNames[values.recipientEmail],
        subject: values.subject,
        body: values.body
      };

      this.emailsService.replyToEmail(this.organisationId, this.email.batchId, this.email.id, data).subscribe(
        () => {
          this.toast.success("Bericht verstuurd.");
          this.toggleExpand();
          this.toggleExpand();
          this.replyExpanded = false;
        },
        () => this.toast.error("Er ging iets mis.")
      );
    }
  }
}
