import { Component, OnInit, OnDestroy } from "@angular/core";
import { FormGroup, FormControl, Validators } from "@angular/forms";
import { TemplatesDataService } from "@app/organisations/templates/templates-data.service";
import { OrganisationsDataService } from "@app/organisations/organisations-data.service";
import { Template, TemplateType } from "@app/shared/models/template.model";
import { DropdownOption, DropdownOptionGroup } from "@app/shared/forms/form-input/form-input.component";
import * as MarkdownIt from "markdown-it";
import { ActivatedRoute, Router } from "@angular/router";
import { EmailsDataService, EmailReply } from "@app/shared/emails/emails-data.service";
import { CompaniesDataService } from "@app/companies/companies-data.service";
import { SchoolsDataService } from "@app/schools/schools-data.service";
import { School } from "@app/shared/models/school.model";
import { Company } from "@app/shared/models/company.model";
import { Subscription, forkJoin, of, Observable } from "rxjs";
import { Entity } from "@app/shared/models/entity.model";
import { tap } from "rxjs/operators";
import { EventsDataService } from "@app/events/events-data.service";
import { Event } from "@app/shared/models/event.model";
import { markFormGroupTouched } from "@app/shared/form-group";
import { NotificationsService } from "angular2-notifications";

@Component({
  selector: "app-email-composer",
  templateUrl: "./email-composer.component.html",
  styleUrls: ["./email-composer.component.scss"]
})
export class EmailComposerComponent implements OnInit, OnDestroy {
  form: FormGroup;
  templates: Template[] = [];
  templateOptions: DropdownOption[] = [];
  showHelp = false;
  currentType: TemplateType;
  variableMap = {};
  md: MarkdownIt = null;

  recipientType = "";
  recipient: Entity;
  recipientContactCount = 0;

  dataSubscription: Subscription;

  companies: Company[];
  schools: School[];
  events: Event[];

  recipientOptions: DropdownOptionGroup[] = [];
  recipients: { type: string; entity: Entity }[] = [];

  constructor(
    private templateService: TemplatesDataService,
    private schoolService: SchoolsDataService,
    private companyService: CompaniesDataService,
    private eventsService: EventsDataService,
    private emailsService: EmailsDataService,
    private organisationsService: OrganisationsDataService,
    private route: ActivatedRoute,
    private router: Router,
    private toast: NotificationsService
  ) {
    this.md = new MarkdownIt();
  }

  ngOnInit() {
    this.dataSubscription = this.route.parent.data.subscribe(data => {
      const organisationId = this.organisationsService.currentOrganisation.id;
      this.recipientType = data["filter"];
      switch (this.recipientType) {
        case "company":
          this.recipient = this.companyService.currentCompany;
          this.companyService
            .getContacts(organisationId, this.recipient.id)
            .subscribe(contacts => (this.recipientContactCount = contacts.filter(c => c.receiveEmail).length));
          break;
        case "school":
          this.recipient = this.schoolService.currentSchool;
          this.schoolService
            .getContacts(organisationId, this.recipient.id)
            .subscribe(contacts => (this.recipientContactCount = contacts.filter(c => c.receiveEmail).length));
          break;
        default:
          const subscriptions = [
            this.companyService
              .getAllForOrganisation(organisationId)
              .pipe(tap(companies => (this.companies = companies))),
            this.schoolService.getAllForOrganisation(organisationId).pipe(tap(schools => (this.schools = schools))),
            this.eventsService.getAllForOrganisation(organisationId).pipe(tap(events => (this.events = events)))
          ];
          forkJoin(subscriptions).subscribe({ complete: this.initRecipients });
          this.recipient = null;
      }

      this.form = new FormGroup({
        subject: new FormControl("", Validators.required),
        body: new FormControl("", Validators.required),
        recipients: new FormControl(""),
        template: new FormControl("")
      });
    });

    this.templateService
      .getAllForOrganisation(this.organisationsService.currentOrganisation.id)
      .subscribe(templatesPerType => {
        this.templates = templatesPerType.find(t => t.type === "general").templates;
        this.templateOptions = this.templates.map(t => new DropdownOption(t.id, t.name));
      });
    this.templateService
      .getVariablesForType(this.organisationsService.currentOrganisation.id, "general")
      .subscribe((typeData: TemplateType) => {
        this.currentType = typeData;
        const variableMapWithoutPercent = {};
        this.variableMap = {};
        this.currentType.variables.global.forEach(v => (variableMapWithoutPercent[v.name] = v.example));
        this.currentType.variables.specific.forEach(v => (variableMapWithoutPercent[v.name] = v.example));
        Object.keys(variableMapWithoutPercent).forEach(
          v => (this.variableMap[`%${v}%`] = variableMapWithoutPercent[v])
        );
      });
  }

  ngOnDestroy() {
    this.dataSubscription.unsubscribe();
  }

  initRecipients = () => {
    this.recipientOptions = [
      new DropdownOptionGroup(
        "Bedrijven",
        [new DropdownOption("allCompanies", "Alle bedrijven")]
          .concat(this.events.map(e => new DropdownOption(`aie-${e.id}`, `Alle deelnemende bedrijven aan ${e.name}`)))
          .concat(this.companies.map(c => new DropdownOption(`com-${c.id}`, c.name)))
      ),
      new DropdownOptionGroup(
        "Scholen",
        [new DropdownOption("allSchools", "Alle scholen")].concat(
          this.schools.map(c => new DropdownOption(`sch-${c.id}`, c.name))
        )
      )
    ];
  };

  recipientsChanged = (event: any) => {
    const options = event.target.options;
    const value = [];
    for (let i = 0, l = options.length; i < l; i++) {
      if (options[i].selected) {
        value.push(options[i].value.replace(/[0-9]+\: \'(.+)/gi, "$1").slice(0, -1));
      }
    }

    this.recipients = value.map(v => {
      let type = v;
      let entity: Entity = null;
      const typeAbbrev = v.slice(0, 3);
      const id = v.slice(4);
      switch (v.slice(0, 3)) {
        case "com":
          type = "company";
          entity = this.companies.find(c => c.id === id);
          break;
        case "sch":
          type = "company";
          entity = this.schools.find(c => c.id === id);
          break;
        case "aie":
          type = "event";
          entity = this.events.find(e => e.id === id);
          break;
      }

      return { type, entity };
    });
  };

  sendEmail = async () => {
    markFormGroupTouched(this.form);

    console.log(this.form);
    if (this.form.valid) {
      let schoolIds: string[] = [];
      let companyIds: string[] = [];
      let events: Observable<Company[]>[] = [];

      if (this.recipient) {
        switch (this.recipientType) {
          case "company":
            companyIds.push(this.recipient.id);
            break;
          case "school":
            schoolIds.push(this.recipient.id);
            break;
        }
      } else {
        companyIds = this.recipients.filter(r => r.type === "company").map(r => r.entity.id);
        schoolIds = this.recipients.filter(r => r.type === "school").map(r => r.entity.id);

        if (this.recipients.find(r => r.type === "allSchools")) {
          schoolIds = this.schools.map(s => s.id);
        }
        if (this.recipients.find(r => r.type === "allCompanies")) {
          companyIds = this.companies.map(c => c.id);
        } else {
          // if not all companies are used, maybe some from an event are.
          const eventsToCheck = this.recipients.filter(r => r.type === "event");
          if (eventsToCheck.length > 0) {
            events = events.concat(
              eventsToCheck.map(e =>
                this.eventsService.getCompaniesForId(this.organisationsService.currentOrganisation.id, e.entity.id)
              )
            );
          }
        }
      }
      // Dummy event to force a complete
      events.push(of([]));
      forkJoin(events).subscribe({
        next: companiesNested =>
          companiesNested.forEach(companies => (companyIds = companyIds.concat(companies.map(c => c.id)))),
        complete: () => {
          // Recipient IDs are initialized, send mail
          const values = this.form.value;

          const data = {
            subject: values.subject,
            body: values.body,
            companyIds,
            schoolIds
          };

          this.emailsService.sendEmail(this.organisationsService.currentOrganisation.id, data).subscribe(
            () => {
              this.toast.success("E-mail succesvol verstuurd.");
              this.router.navigate(["../"], { relativeTo: this.route });
            },
            e => this.toast.error("Er ging iets mis.", e.error.message)
          );
        }
      });
    }
  };

  getMarkdownParsedValue = () => {
    let parsedValue = this.md.render(this.form.get("body").value);

    if (Object.keys(this.variableMap).length > 0) {
      const regEx = new RegExp(Object.keys(this.variableMap).join("|"), "gi");
      parsedValue = parsedValue.replace(regEx, (matched: string) => this.variableMap[matched]);
    }
    return parsedValue;
  };

  loadTemplate() {
    const template = this.templates.find(t => t.id === this.form.get("template").value);
    this.form.get("subject").setValue(template.subject);
    this.form.get("body").setValue(template.body);
  }
}
