import { Component, OnInit } from "@angular/core";
import { FormControl, FormGroup, Validators } from "@angular/forms";
import { Organisation } from "@app/shared/models/organisation.model";
import { CrudComponent } from "@app/shared/crud.component";
import { Entity } from "@app/shared/models/entity.model";
import { Router } from "@angular/router";
import { FormModalComponent, FormModalInputOptions } from "@app/shared/form-modal/form-modal.component";
import { OrganisationsDataService } from "@app/organisations/organisations-data.service";
import { distinctUntilChanged } from "rxjs/internal/operators";
import { NgbModal } from "@ng-bootstrap/ng-bootstrap";

@Component({
  selector: "app-organisations-list",
  templateUrl: "./organisations-list.component.html",
  styleUrls: ["./organisations-list.component.scss"],
})
export class OrganisationsListComponent extends CrudComponent<Organisation> implements OnInit {
  constructor(modal: NgbModal, private router: Router, private organisationsService: OrganisationsDataService) {
    super();
    super.initialize(modal);
  }

  ngOnInit() {
    this.organisationsService
      .getAll()
      .pipe(distinctUntilChanged())
      .subscribe((organisations: Organisation[]) => {
        this.entities = organisations;
        if (organisations.length === 1) {
          this.rowClicked({ selected: [organisations[0]] });
        }
      });
  }

  protected createForm() {
    this.crudForm = new FormGroup({
      name: new FormControl("", [Validators.required]),
    });
  }

  rowClicked(event: any) {
    this.router.navigate(["/organisaties", event.selected[0].id, "evenementen"]);
  }

  editPressed(entity: Entity) {
    const org = entity as Organisation;

    const modalRef = this.modalService.open(FormModalComponent);
    const component = modalRef.componentInstance as FormModalComponent;
    component.title = "Organisatie aanpassen";
    component.confirmLabel = "Aanpassen";
    component.fields = [
      new FormModalInputOptions("name", "Organisatienaam", "text", org.name, [Validators.required]),
      new FormModalInputOptions("primaryColor", "Themakleur", "color", org.primaryColor, [Validators.required]),
      new FormModalInputOptions("slug", "URL slug", "text", org.slug, [
        Validators.required,
        Validators.pattern(/[a-z\-]+/),
      ]),
    ];
    component.callback = (result: any) =>
      this.organisationsService.update(entity.id, {
        name: result.name,
        primaryColor: result.primaryColor,
        slug: result.slug,
        id: entity.id,
      });
  }

  detailPressed(entity: Entity) {
    this.router.navigate(["/organisaties", entity.id]);
  }

  newPressed() {
    const modalRef = this.modalService.open(FormModalComponent);
    const component = modalRef.componentInstance as FormModalComponent;
    component.title = "Nieuwe organisatie toevoegen";
    component.confirmLabel = "Toevoegen";
    component.fields = [
      new FormModalInputOptions("name", "Organisatienaam", "text", "", [Validators.required]),
      new FormModalInputOptions("primaryColor", "Themakleur", "color", "#008cd6", [Validators.required]),
    ];
    component.callback = (result: any) => this.organisationsService.add(result);
  }
}
