import {Component, OnInit} from '@angular/core';
import {Validators} from '@angular/forms';
import {NgbModal} from '@ng-bootstrap/ng-bootstrap';
import {CrudComponent} from '@app/shared/crud.component';
import {FormModalComponent, FormModalInputOptions} from '@app/shared/form-modal/form-modal.component';
import {Location} from '@app/shared/models/location.model';
import {Address} from '@app/shared/models/address.model';
import {Title} from '@angular/platform-browser';
import {take, tap} from 'rxjs/internal/operators';
import {LocationsDataService} from '@app/organisations/locations/locations-data.service';
import {OrganisationsDataService} from "@app/organisations/organisations-data.service";

@Component({
  selector: 'app-locations',
  templateUrl: './organisation-locations.component.html',
  styleUrls: ['./organisation-locations.component.scss']
})
export class OrganisationLocationsComponent extends CrudComponent<Location> implements OnInit {
  organisationId: string;

  constructor(modal: NgbModal,
              private locationsService: LocationsDataService,
              private organisationsService: OrganisationsDataService,
              private title: Title) {
    super();
    super.initialize(modal);
  }

  ngOnInit() {
    this.organisationId = this.organisationsService.currentOrganisation.id;
    this.title.setTitle(`Organisaties > ${this.organisationsService.currentOrganisation.name} > Locaties`);
    this.loadLocations();
  }

  loadLocations() {
    this.locationsService.getAllForOrganisation(this.organisationId).pipe(take(1))
      .subscribe(locs => this.entities = locs);
  }

  rowClicked(event: any) {
    const location = event.selected[0] as Location;

    const modalRef = this.modalService.open(FormModalComponent);
    const component = modalRef.componentInstance as FormModalComponent;
    component.title = 'Locatie aanpassen';
    component.confirmLabel = 'Aanpassen';
    component.fields = [
      new FormModalInputOptions('name', 'Naam van de locatie',
        'text', location.name, [Validators.required]),
      new FormModalInputOptions('description', 'Beschrijving',
        'text', location.description || "", [Validators.required]),
      new FormModalInputOptions('postcode', 'Postcode',
        'text', location.address ? location.address.postcode : '',
        [Validators.pattern(/^( )*[0-9]{4}[a-zA-Z]{2}( )*$/)]),
      new FormModalInputOptions('number', 'Huisnummer',
        'text', location.address ? location.address.number.toString() : '', [Validators.pattern(/^[0-9]+$/)]),
    ];

    component.callback = (result) => {
      if (result.postcode || result.number) {
        return this.locationsService.updateInOrganisation(this.organisationId, {
          id: location.id,
          ...location,
          name: result.name,
          description: result.description,
          address: {...location.address, number: parseInt(result.number), postcode: result.postcode}
        }).pipe(tap(() => this.loadLocations()));
      } else {
        return this.locationsService.updateInOrganisation(this.organisationId, {
          id: location.id,
          ...location,
          name: result.name,
          description: result.description
        }).pipe(tap(() => this.loadLocations()));
      }
    };
  }

  newPressed() {
    const modalRef = this.modalService.open(FormModalComponent);
    const component = modalRef.componentInstance as FormModalComponent;

    component.callback = (result) => {
      if (result.postcode || result.number) {
        return this.locationsService.addToOrganisation(this.organisationId, {
          name: result.name,
          description: result.description,
          address: {...new Address(), number: parseInt(result.number), postcode: result.postcode}
        }).pipe(tap(() => this.loadLocations()));
      } else {
        return this.locationsService.addToOrganisation(this.organisationId, {
          name: result.name,
          description: result.description
        }).pipe(tap(() => this.loadLocations()));
      }
    };
    component.title = 'Nieuwe locatie toevoegen';
    component.confirmLabel = 'Toevoegen';
    component.fields = [
      new FormModalInputOptions('name', 'Naam van de locatie',
        'text', '', [Validators.required]),
      new FormModalInputOptions('description', 'Beschrijving',
        'text', '', [Validators.required]),
      new FormModalInputOptions('postcode', 'Postcode',
        'text', '', [Validators.pattern(/^( )*[0-9]{4}[a-zA-Z]{2}( )*$/)]),
      new FormModalInputOptions('number', 'Huisnummer',
        'text', '', [Validators.pattern(/^[0-9]+$/)]),
    ];
  }
}
