import { Component, OnInit } from "@angular/core";
import { Validators } from "@angular/forms";
import { NgbModal } from "@ng-bootstrap/ng-bootstrap";
import { FormModalComponent, FormModalInputOptions } from "@app/shared/form-modal/form-modal.component";
import { NotificationsService } from "angular2-notifications";
import { Title } from "@angular/platform-browser";
import { OrganisationsDataService } from "@app/organisations/organisations-data.service";
import { Member } from "@app/shared/models/member.model";
import { DeleteModalComponent } from "@app/shared/delete-modal/modal.component";
import { DropdownOption } from "@app/shared/forms/form-input/form-input.component";
import { Router } from "@angular/router";

@Component({
  selector: "app-organisations-members",
  templateUrl: "./organisations-members.component.html",
  styleUrls: ["./organisations-members.component.scss"]
})
export class OrganisationsMembersComponent implements OnInit {
  organisationId = "";
  tempAdmins: Member[] = [];
  tempGuides: Member[] = [];
  selected: Member[] = [];
  admins: Member[] = [];
  guides: Member[] = [];

  constructor(
    private organisationsService: OrganisationsDataService,
    private modal: NgbModal,
    private toast: NotificationsService,
    private title: Title,
    private router: Router
  ) {}

  ngOnInit() {
    this.organisationId = this.organisationsService.currentOrganisation.id;
    this.title.setTitle(`Organisaties > ${this.organisationsService.currentOrganisation.name} > Accounts`);

    this.loadMembers();
  }

  rowClicked(event: any) {
    this.router.navigate(["/organisaties", this.organisationId, "accounts", event.selected[0].id]);
  }

  loadMembers(resetTemp: boolean = true) {
    this.organisationsService.getMembers(this.organisationId).subscribe(members => {
      this.admins = members.filter(m => m.accessLevel === "organisationAdmin");
      this.guides = members.filter(m => m.accessLevel === "organisationGuide");
      if(resetTemp) {
        this.tempAdmins = this.admins;
        this.tempGuides = this.guides;
      }
    });
  }

  newPressed() {
    const modalRef = this.modal.open(FormModalComponent);
    const component = modalRef.componentInstance as FormModalComponent;
    component.callback = result => {
      return this.organisationsService.addMember(this.organisationId, result, Number.parseInt(result.accessLevelId));
    };
    component.onSuccess = () => {
      this.toast.success("Gebruiker succesvol toegevoegd.");
      this.loadMembers(false);
    };
    component.onError = err => {
      this.toast.error("Er ging iets mis.", err.error.message);
    };
    component.title = "Nieuw account toevoegen";
    component.confirmLabel = "Toevoegen";
    component.fields = [
      new FormModalInputOptions("firstName", "Voornaam", "text", "", [Validators.required]),
      new FormModalInputOptions("insertion", "Tussenvoegsel", "text", ""),
      new FormModalInputOptions("lastName", "Achternaam", "text", "", [Validators.required]),
      new FormModalInputOptions("username", "Gebruikersnaam", "text", "", [Validators.required]),
      new FormModalInputOptions(
        "accessLevelId",
        "Rol",
        "select",
        {
          value: 1,
          options: [
            { id: 1, name: "Beheerder" },
            { id: 3, name: "Begeleider" }
          ].map(type => new DropdownOption(type.id, type.name))
        },
        [Validators.required]
      )
    ];
  }

  deletePressed(entity: Member) {
    const modalRef = this.modal.open(DeleteModalComponent);
    modalRef.componentInstance.name = entity.username;
    modalRef.result
      .then(result => {
        this.organisationsService.deleteMember(this.organisationId, entity).subscribe(
          member => {
            this.toast.success(`Beheerder ${entity.username} succesvol verwijderd.`);
            this.loadMembers(false);
          },
          () => this.toast.error(`Er ging iets mis.`)
        );
      })
      .catch(() => {});
  }

  updateFilter(event: any) {
    const val = event.target.value.toLowerCase();

    // filter our data
    const tempAdmins = this.tempAdmins.filter(d => {
      return (
        JSON.stringify(d)
          .toLowerCase()
          .indexOf(val) !== -1 || !val
      );
    });
    const tempGuides = this.tempGuides.filter(d => {
      return (
        JSON.stringify(d)
          .toLowerCase()
          .indexOf(val) !== -1 || !val
      );
    });

    // update the rows
    this.admins = tempAdmins;
    this.guides = tempGuides;
  }
}
