import {Component, OnInit} from '@angular/core';
import {FormControl, FormGroup, Validators} from '@angular/forms';
import {NgbModal} from '@ng-bootstrap/ng-bootstrap';
import {CrudComponent} from '@app/shared/crud.component';
import {NotificationsService} from 'angular2-notifications';
import {Title} from '@angular/platform-browser';
import {take} from 'rxjs/internal/operators';
import {Message} from '@app/shared/models/message.model';
import {MessagesDataService} from '@app/organisations/messages/messages-data.service';
import {markFormGroupTouched} from '@app/shared/form-group';
import {Organisation} from '@app/shared/models/organisation.model';
import {RecipientModalComponent} from '@app/organisations/messages/recipient-modal/recipient-modal.component';
import {forkJoin, of} from 'rxjs';
import {School} from '@app/shared/models/school.model';
import {Company} from '@app/shared/models/company.model';
import {CompaniesDataService} from '@app/companies/companies-data.service';
import {SchoolsDataService} from '@app/schools/schools-data.service';
import {DeleteModalComponent} from '@app/shared/delete-modal/modal.component';
import {OrganisationsDataService} from "@app/organisations/organisations-data.service";
import { AuthDataService } from '../../shared/auth-data.service';

@Component({
  selector: 'app-messages',
  templateUrl: './messages.component.html',
  styleUrls: ['./messages.component.scss']
})
export class MessagesComponent extends CrudComponent<Message> implements OnInit {
  senderTypeOptions: any = [
    {title: 'Organisaties', options: []},
    {title: 'Gebruikers', options: []},
    {title: 'Scholen', options: []},
    {title: 'Bedrijven', options: []},
  ];

  companies: Company[];
  schools: School[];

  sending = false;
  organisation: Organisation;

  newMessageForm: FormGroup;
  recipients: any[] = [];

  edittingMessage: Message = null;

  constructor(modal: NgbModal, private toast: NotificationsService,
              private messagesService: MessagesDataService,
              private title: Title,
              private organisationsService: OrganisationsDataService,
              private companiesService: CompaniesDataService,
              private schoolsService: SchoolsDataService,
              private authDataService: AuthDataService) {
    super();
    super.initialize(modal);
  }

  ngOnInit() {
    this.newMessageForm = new FormGroup({
      senderType: new FormControl('organisation', Validators.required),
      sendAt: new FormControl(null, []),
      subject: new FormControl('', Validators.required),
      body: new FormControl('', Validators.required)
    });

    this.senderTypeOptions[1].options = [
      {value: 'user', label: this.authDataService.user.username}
    ];

    this.organisation = this.organisationsService.currentOrganisation;

    this.senderTypeOptions[0].options = [
      {value: 'organisation', label: this.organisation.name}
    ];

    this.schoolsService.getAllForOrganisation(this.organisation.id).subscribe(schools => {
      this.schools = schools;
      this.senderTypeOptions[2].options = this.schools.map(s => {
        return {label: s.name, value: s.id};
      });
    });
    this.companiesService.getAllForOrganisation(this.organisation.id).subscribe(companies => {
      this.companies = companies;
      this.senderTypeOptions[3].options = this.companies.map(c => {
        return {label: c.name, value: c.id};
      });
    });

    this.messagesService.getAllForOrganisation(this.organisation.id).pipe(take(1)).subscribe(messages => this.entities = messages);
    this.title.setTitle(`Organisaties > ${this.organisation.name} > Berichten`);
  }

  onAddRecipientPressed() {
    const sender = {};
    const senderType = this.newMessageForm.get('senderType').value;
    if (senderType !== 'user' && senderType !== 'organisation') {
      const school = this.schools.find(s => s.id === senderType);
      if (school) {
        sender['type'] = 'school';
        sender['entity'] = school;
      } else {
        const company = this.companies.find(c => c.id === senderType);
        sender['type'] = 'company';
        sender['entity'] = company;
      }
    } else {
      sender['type'] = senderType;
    }

    const modalRef = this.modalService.open(RecipientModalComponent, {size: 'lg'});
    const component = modalRef.componentInstance as RecipientModalComponent;
    component.organisation = this.organisation;
    component.sender = sender;
    component.callback = (result) => {
      this.recipients.push(result);
      return of(result);
    };
  }

  rowClicked(event: any) {
    this.messagesService.getByIdForOrganisation(this.organisation.id, event.selected[0].id).subscribe(message => {
      this.edittingMessage = message;
      this.newMessageForm.reset(this.edittingMessage);
      const sender = this.edittingMessage.sender.type;
      this.newMessageForm.get('senderType').setValue((sender === 'organisation' || sender === 'user')
        ? sender : this.edittingMessage.sender.id);
      this.recipients = [...this.edittingMessage.recipients];
      console.log(this.edittingMessage.recipients);
    });
  }

  arraysEqual(arr1: any[], arr2: any[]) {
    if (arr1.length !== arr2.length) {
      return false;
    }
    for (let i = arr1.length; i--;) {
      if (arr1[i] !== arr2[i]) {
        return false;
      }
    }

    return true;
  }

  onSubmit() {
    markFormGroupTouched(this.newMessageForm);

    if (this.newMessageForm.valid) {
      const message = {...this.newMessageForm.value, recipients: this.recipients};
      if (message.sendAt < Date.now()) {
        message.sendAt = undefined;
      }

      const senderType = this.newMessageForm.get('senderType').value;
      if (senderType !== 'user' && senderType !== 'organisation') {
        const school = this.schools.find(s => s.id === senderType);
        if (school) {
          message.schoolId = school.id;
          message.senderType = 'school';
        } else {
          const company = this.companies.find(c => c.id === senderType);
          message.companyId = company.id;
          message.senderType = 'company';
        }
      }

      this.sending = true;
      if (this.edittingMessage) {

        if (!this.arraysEqual(this.recipients, this.edittingMessage.recipients)) {
          // update recipients
          const recipientsToAdd: any[] = [];
          const recipientsToRemove: any[] = [];
          this.edittingMessage.recipients.forEach(recipient => {
            const idToCheck = MessagesDataService.idFromRecipient(recipient);

            if (this.recipients.findIndex(r => {
              const idToCheck2 = MessagesDataService.idFromRecipient(r);
              return idToCheck2 === idToCheck;
            }) === -1) { // removed
              recipientsToRemove.push(recipient);
            }
          });

          this.recipients.forEach(recipient => {
            const idToCheck = MessagesDataService.idFromRecipient(recipient);

            if (this.edittingMessage.recipients.findIndex(r => {
              const idToCheck2 = MessagesDataService.idFromRecipient(r);
              return idToCheck2 === idToCheck;
            }) === -1) { // added
              recipientsToAdd.push(recipient);
            }
          });

          const addCalls = forkJoin(recipientsToAdd.map(r =>
            this.messagesService.addRecipientToMessage(this.organisation.id, this.edittingMessage.id, r)));
          const removeCalls = forkJoin(recipientsToRemove.map(r =>
            this.messagesService.removeRecipientFromMessage(this.organisation.id, this.edittingMessage.id, r)));
          addCalls.subscribe();
          removeCalls.subscribe();
        }

        this.messagesService.updateInOrganisation(this.organisation.id,
          {id: this.edittingMessage.id, ...message}).subscribe(() => {

          this.toast.success('Bericht aangepast');
          this.sending = false;
          this.messagesService.getAllForOrganisation(this.organisation.id).pipe(take(1))
            .subscribe(messages => this.entities = messages);
        }, err => {
          this.toast.error('Er ging iets mis', err.error.message);
          this.sending = false;
        });
      } else {
        this.messagesService.addToOrganisation(this.organisation.id, message).subscribe(() => {
          this.toast.success('Bericht opgeslagen');
          this.sending = false;
          this.messagesService.getAllForOrganisation(this.organisation.id).pipe(take(1))
            .subscribe(messages => this.entities = messages);
        }, err => {
          this.toast.error('Er ging iets mis', err.error.message);
          this.sending = false;
        });
      }
    }
  }

  getSendLabel() {
    return (this.newMessageForm.get('sendAt').value ? 'Inplannen' : 'Opslaan') +
      (this.sending ? '<div class="loader"></div>' : '');
  }

  removeRecipient(index: number) {
    this.recipients.splice(index, 1);
  }

  getSenderLabel() {
    const senderType = this.newMessageForm.get('senderType').value;
    switch (senderType) {
      case 'organisation':
        return this.senderTypeOptions[0].options[0].label;
      case 'user':
        return this.senderTypeOptions[1].options[0].label;
      default:
        const school = this.schools.find(s => s.id === senderType);
        if (school) {
          return school.name;
        } else {
          return this.companies.find(c => c.id === senderType).name;
        }
    }
  }

  sendPressed(entity: Message) {
    this.messagesService.send(this.organisation.id, entity).subscribe((response) => {
      console.log(response);
      this.toast.success('Succesvol verzonden.');
    }, (err) => this.toast.error('Er ging iets mis.'));
  }

  deletePressed(entity: Message) {
    const modalRef = this.modalService.open(DeleteModalComponent);
    modalRef.componentInstance.name = `Bericht "${entity.subject}"`;
    modalRef.result.then((result) => {
      this.messagesService.deleteFromOrganisation(this.organisation.id, entity.id).subscribe((response) => {
        this.toast.success('Succesvol Verwijderd.');
        this.messagesService.getAllForOrganisation(this.organisation.id).pipe(take(1)).subscribe(messages => this.entities = messages);
      }, (err) => this.toast.error('Er ging iets mis.', err.error.message));
    }).catch(() => {
    });
  }

  cancelEdit() {
    this.edittingMessage = null;
    this.newMessageForm.reset();
    this.newMessageForm.get('senderType').setValue('organisation');
    this.recipients = [];
  }
}
