import { Component, Input, OnInit } from "@angular/core";
import { ValidatorFn } from "@angular/forms";
import { CompaniesDataService } from "@app/companies/companies-data.service";
import { EventsDataService } from "@app/events/events-data.service";
import { OrganisationsDataService } from "@app/organisations/organisations-data.service";
import { SchoolsDataService } from "@app/schools/schools-data.service";
import { Company } from "@app/shared/models/company.model";
import { Event } from "@app/shared/models/event.model";
import { Group } from "@app/shared/models/group.model";
import { Member } from "@app/shared/models/member.model";
import { Organisation } from "@app/shared/models/organisation.model";
import { School } from "@app/shared/models/school.model";
import { NgbActiveModal } from "@ng-bootstrap/ng-bootstrap";
import { forkJoin, from, of } from "rxjs";
import { flatMap } from "rxjs/internal/operators";

export class FormModalInputOptions {
  constructor(
    public id: string,
    public label: string,
    public type: string = "text",
    public value: any = null,
    public validators: ValidatorFn[] = [],
    public extra: any = null
  ) {}
}

@Component({
  selector: "app-recipient-modal",
  templateUrl: "./recipient-modal.component.html",
  styleUrls: ["./recipient-modal.component.scss"]
})
export class RecipientModalComponent implements OnInit {
  @Input() title = "Ontvanger(s) toevoegen";
  @Input() confirmLabel = "Toevoegen";

  sender: any;
  companies: Company[];
  events: Event[];
  schools: School[];
  members: Member[];
  groups: Group[] = [];

  selectedEventIdForGroup = "";

  @Input() organisation: Organisation;

  @Input() callback = (value: any) => {};

  constructor(
    public activeModal: NgbActiveModal,
    private eventsService: EventsDataService,
    private organisationsService: OrganisationsDataService,
    private companiesService: CompaniesDataService,
    private schoolsService: SchoolsDataService
  ) {}

  ngOnInit() {
    this.companiesService
      .getAllForOrganisation(this.organisation.id)
      .subscribe(companies => (this.companies = companies));
    this.eventsService.getAllForOrganisation(this.organisation.id).subscribe(events => {
      this.events = events.reverse();
      this.setSelectedEventForGroup(this.events[0].id);

      if (this.sender.type !== "school") {
        this.schoolsService.getAllForOrganisation(this.organisation.id).subscribe(schools => {
          this.schools = schools;
          this.groups = [];
          from(this.schools)
            .pipe(
              flatMap(s => of(s)),
              flatMap(s =>
                forkJoin(
                  this.events.map(e => this.schoolsService.getAllGroupsForSchool(this.organisation.id, s.id, e.id))
                )
              ),
              flatMap(g => g)
            )
            .subscribe(groups => {
              this.groups.push(...groups);
            });
        });
      } else {
        this.schools = [this.sender.entity];
        forkJoin(
          this.events.map(e =>
            this.schoolsService.getAllGroupsForSchool(this.organisation.id, this.sender.entity.id, e.id)
          )
        )
          .pipe(flatMap(g => g))
          .subscribe(groups => (this.groups = groups));

        this.schoolsService.getAllGroupsForSchool(this.organisation.id, this.schools[0].id);
      }
    });
    this.organisationsService.getAllMembers(this.organisation.id).subscribe(members => (this.members = members));
  }

  onSubmit(value: any) {
    this.callback(value);
  }

  onSubmitEventAndMaybeCompany(eventsSelect: any, companiesSelect: any) {
    const eventId = eventsSelect.value;
    const event = this.events.find(e => e.id === eventId);

    const companyId = companiesSelect.value;
    if (companyId === "") {
      this.onSubmit({ name: event.name, eventId: event.id, type: "event" });
      return;
    }
    const company = this.companies.find(c => c.id === companyId);
    this.onSubmit({
      name: company.name + " tijdens " + event.name,
      companyId: company.id,
      eventId: event.id,
      type: "visitors"
    });
  }

  onSubmitSchoolAndMaybeEvent(schoolsSelect: any, eventsSelect: any) {
    const schoolId = schoolsSelect.value;
    const school = this.schools.find(s => s.id === schoolId);

    const eventId = eventsSelect.value;
    if (eventId === "") {
      this.onSubmit({ name: school.name, schoolId: school.id, type: "school" });
      return;
    }
    const event = this.events.find(e => e.id === eventId);
    this.onSubmit({
      name: event.name + ` (alleen ${school.name})`,
      schoolId: school.id,
      eventId: event.id,
      type: "event"
    });
  }

  onSubmitUser(userSelect: any) {
    const userId = userSelect.value;
    const user = this.members.find(u => u.id === userId);

    this.onSubmit({ name: user.username, userId: user.id, type: "user" });
  }

  onSubmitGroup(groupSelect: any) {
    const groupId = groupSelect.value;
    const group = this.groups.find(g => g.id === groupId);

    this.onSubmit({ name: group.name, groupId: group.id, type: "group" });
  }

  groupsForEvent(eventId: string) {
    return this.groups.filter(g => g.eventId === eventId);
  }

  setSelectedEventForGroup(eventId: string) {
    this.selectedEventIdForGroup = eventId;
  }
}
