import { EventEmitter, Injectable } from "@angular/core";
import { HttpClient } from "@angular/common/http";

import { Observable } from "rxjs";

import { Organisation } from "@app/shared/models/organisation.model";
import { Member } from "@app/shared/models/member.model";
import { Location } from "../shared/models/location.model";
import { tap } from "rxjs/operators";

@Injectable()
export class OrganisationsDataService {
  organisations: Organisation[] = [];
  members: { [organisationId: string]: Member[] } = {};
  locations: { [organisationId: string]: Location[] } = {};
  _currentOrganisation: Organisation = null;
  organisationChanged = new EventEmitter();

  get currentOrganisation(): Organisation {
    return this._currentOrganisation;
  }

  set currentOrganisation(value: Organisation) {
    this._currentOrganisation = value;
    this.organisationChanged.emit(value);
  }

  constructor(private http: HttpClient) {}

  getAll(): Observable<Organisation[]> {
    return this.http
      .get<Organisation[]>(`/organisations/`)
      .pipe(tap(organisations => (this.organisations = organisations)));
  }

  getById(key: string | number): Observable<Organisation> {
    return this.http.get<Organisation>(`/organisations/${key}`);
  }

  update(organisationId: string, entity: Partial<Organisation>) {
    return this.http.put<Organisation>(`/organisations/${organisationId}`, entity);
  }

  add(entity: Organisation) {
    return this.http.post<Organisation>(`/organisations/`, entity);
  }

  getMembers(organisationId: string): Observable<Member[]> {
    return this.http.get<Member[]>(`/organisations/${organisationId}/members?accessLevels=[1,3]`).pipe(
      tap(members => {
        this.members[organisationId] = (this.members[organisationId] || []).concat(
          members.filter(
            (m1: Member) =>
              !this.members[organisationId] ||
              this.members[organisationId].findIndex((m2: Member) => m1.id === m2.id) === -1
          )
        );
      })
    );
  }

  getAllMembers(organisationId: string): Observable<Member[]> {
    return this.http
      .get<Member[]>(`/organisations/${organisationId}/members`)
      .pipe(tap(members => (this.members[organisationId] = members)));
  }

  addMember(
    organisationId: string,
    data: { username: string; firstName: string; insertion: string; lastName: string },
    accessLevelId: number = 1
  ) {
    return this.http
      .post(`/organisations/${organisationId}/members`, {
        username: data.username,
        accessLevelId,
        person: { firstName: data.firstName, insertion: data.insertion || null, lastName: data.lastName }
      })
      .pipe(tap(() => this.getAllMembers(organisationId)));
  }

  deleteMember(organisationId: string, member: Member) {
    return this.http
      .delete(`/organisations/${organisationId}/members/${member.id}`)
      .pipe(tap(() => this.getAllMembers(organisationId)));
  }

  updateMember(organisationId: string, data: any): Observable<Member> {
    return this.http
      .put<Member>(`/organisations/${organisationId}/members/${data.id}`, data)
      .pipe(tap(() => this.getAllMembers(organisationId)));
  }

  getLocations(organisationId: string): Observable<Location[]> {
    return this.http.get<Location[]>(`/organisations/${organisationId}/locations`).pipe(
      tap((locations: Location[]) => {
        this.locations[organisationId] = (this.locations[organisationId] || []).concat(
          locations.filter(
            (m1: Location) =>
              !this.locations[organisationId] ||
              this.locations[organisationId].findIndex((m2: Location) => m1.id === m2.id) === -1
          )
        );
      })
    );
  }
}
