import { Component, OnInit } from "@angular/core";
import { FormGroup, FormControl, Validators } from "@angular/forms";
import * as Markdown from "markdown-it";
import { Template, TemplateType, TemplateTypeVariable, templateTypeOptions } from "@app/shared/models/template.model";
import { TemplatesDataService, TemplatesPerType } from "./templates-data.service";
import { OrganisationsDataService } from "../organisations-data.service";
import { markFormGroupTouched } from "@app/shared/form-group";
import { DeleteModalComponent } from "@app/shared/delete-modal/modal.component";
import { NgbModal } from "@ng-bootstrap/ng-bootstrap";
import { NotificationsService } from "angular2-notifications";

@Component({
  selector: "app-templates",
  templateUrl: "./templates.component.html",
  styleUrls: ["./templates.component.scss"]
})
export class TemplatesComponent implements OnInit {
  showHelp = false;
  form: FormGroup;
  md: any;
  templatesPerType: TemplatesPerType[] = [];
  templates: Template[] = [];
  editting: Template = null;
  variables: any;
  typeOptions = [...templateTypeOptions];
  currentType: TemplateType;
  variableMap = {};

  constructor(
    private organisationsService: OrganisationsDataService,
    private templatesService: TemplatesDataService,
    private toast: NotificationsService,
    private modalService: NgbModal
  ) {
    this.md = new Markdown();
  }

  ngOnInit() {
    this.initialize();
  }

  initialize() {
    this.form = new FormGroup({
      name: new FormControl("", [Validators.required]),
      subject: new FormControl("", [Validators.required]),
      type: new FormControl("general", [Validators.required]),
      body: new FormControl("", [Validators.required]),
      primary: new FormControl(true)
    });

    this.initializeType("general");

    this.form.controls["type"].valueChanges.subscribe(value => this.initializeType(value));

    this.templatesService
      .getAllForOrganisation(this.organisationsService.currentOrganisation.id)
      .subscribe((templatesPerType: TemplatesPerType[]) => {
        this.templatesPerType = templatesPerType;
        this.templates = [];
        this.templatesPerType.map(tpt => (this.templates = this.templates.concat(tpt.templates)));
      });
  }

  initializeType(type: string) {
    this.templatesService
      .getVariablesForType(this.organisationsService.currentOrganisation.id, type)
      .subscribe((typeData: TemplateType) => {
        this.currentType = typeData;
        const variableMapWithoutPercent = {};
        this.variableMap = {};
        this.currentType.variables.global.forEach(v => (variableMapWithoutPercent[v.name] = v.example));
        this.currentType.variables.specific.forEach(v => (variableMapWithoutPercent[v.name] = v.example));
        Object.keys(variableMapWithoutPercent).forEach(
          v => (this.variableMap[`%${v}%`] = variableMapWithoutPercent[v])
        );
      });
  }

  getMarkdownParsedValue = () => {
    let parsedValue = this.md.render(this.form.get("body").value);

    if (Object.keys(this.variableMap).length > 0) {
      const regEx = new RegExp(Object.keys(this.variableMap).join("|"), "gi");
      parsedValue = parsedValue.replace(regEx, (matched: string) => this.variableMap[matched]);
    }
    return parsedValue;
  };

  onSubmit() {
    markFormGroupTouched(this.form);
    if (this.form.valid) {
      if (this.editting) {
        const template = { ...this.editting, ...this.form.value };
        this.templatesService
          .updateInOrganisation(this.organisationsService.currentOrganisation.id, template)
          .subscribe(
            () => {
              this.toast.success("Template succesvol aangepast");
              this.initialize();
            },
            error => {
              if (error.error.name === "MissingVariableError") {
                this.toast.error(
                  `Er ging iets mis.`,
                  `De volgende variabelen ontbreken: ${error.error.data.missing
                    .map((m: string) => `%${m}%`)
                    .join(", ")}`
                );
              } else {
                this.toast.error(`Er ging iets mis.", ${error.error.message}"`);
              }
            }
          );
      } else {
        const template = { ...this.form.value };
        this.templatesService.addToOrganisation(this.organisationsService.currentOrganisation.id, template).subscribe(
          () => {
            this.toast.success("Template succesvol opgeslagen");
            this.initialize();
          },
          error => {
            if (error.error.name === "MissingVariableError") {
              this.toast.error(
                `Er ging iets mis.`,
                `De volgende variabelen ontbreken: ${error.error.data.missing.join(", ")}`
              );
            } else {
              this.toast.error(`Er ging iets mis.", ${error.error.message}"`);
            }
          }
        );
      }
    } else {
      this.toast.error("Niet alle velden zijn juist ingevuld.");
    }
  }

  deletePressed(row: Template) {
    const modalRef = this.modalService.open(DeleteModalComponent);
    const component = modalRef.componentInstance;
    component.name = row.name;

    modalRef.result
      .then(result => {
        this.templatesService
          .deleteFromOrganisation(this.organisationsService.currentOrganisation.id, row.id)
          .subscribe(() => {
            this.toast.success("Template verwijderd");
            this.initialize();
          });
      })
      .catch(() => {});
  }

  editPressed(row: Template) {
    this.form.reset(row);
    this.editting = row;
  }

  cancel() {
    this.editting = null;
    this.form.reset({ body: "", type: "accountCreated" });
  }

  saveEnabled() {
    return (!this.editting || !this.editting.autoGenerated || !this.editting.primary) && this.form.valid;
  }
}
