import { FormModalComponent, FormModalInputOptions } from "./../../shared/form-modal/form-modal.component";
import { NgbModal } from "@ng-bootstrap/ng-bootstrap";
import { Component, OnInit } from "@angular/core";
import * as Color from "color";
import { markFormGroupTouched } from "@app/shared/form-group";
import { FormControl, FormGroup, Validators } from "@angular/forms";
import { NotificationsService } from "angular2-notifications";
import { take } from "rxjs/internal/operators";
import { Title } from "@angular/platform-browser";
import { Contact } from "@app/shared/models/contact.model";
import { School } from "@app/shared/models/school.model";
import { SchoolsDataService } from "@app/schools/schools-data.service";
import { OrganisationsDataService } from "@app/organisations/organisations-data.service";
import { Organisation } from "@app/shared/models/organisation.model";

@Component({
  selector: "app-school-detail",
  templateUrl: "./school-detail.component.html",
  styleUrls: ["./school-detail.component.scss"]
})
export class SchoolDetailComponent implements OnInit {
  forms = {
    general: new FormGroup({})
  };

  organisation: Organisation = null;
  school: School = null;
  color: Color;

  contacts: Contact[] = [];
  selectedContacts: Contact[] = [];

  constructor(
    private organisationsService: OrganisationsDataService,
    private schoolsService: SchoolsDataService,
    private modalService: NgbModal,
    private toast: NotificationsService,
    private title: Title
  ) {}

  ngOnInit() {
    this.organisation = this.organisationsService.currentOrganisation;
    this.initialize(this.schoolsService.currentSchool);
  }

  initialize(school: School) {
    this.title.setTitle(`Organisaties > ${this.organisation.name} > Scholen > ${school.name}`);
    if (!school) {
      return;
    }

    this.school = school;
    this.color = Color("#eee");

    this.forms.general = new FormGroup({
      name: new FormControl(school.name, Validators.required),
      prefix: new FormControl({
        value: school.prefix,
        disabled: true
      }),
      postcode: new FormControl(school.address ? school.address.postcode : "", [
        Validators.pattern(/^( )*[0-9]{4}( )*[a-zA-Z]{2}( )*$/)
      ]),
      number: new FormControl(school.address ? school.address.number : "", [Validators.pattern(/^[0-9]+$/)]),
      numberAddition: new FormControl(school.address ? school.address.numberAddition : ""),
      street: new FormControl({
        value: school.address ? school.address.street : "",
        disabled: true
      }),
      city: new FormControl({
        value: school.address ? school.address.city : "",
        disabled: true
      })
    });

    // Load contacts
    this.schoolsService
      .getContacts(this.organisation.id, this.school.id)
      .subscribe(contacts => (this.contacts = contacts));
  }

  updateGeneralInfo() {
    markFormGroupTouched(this.forms.general);
    const { number, postcode, numberAddition, name } = this.forms.general.value;
    const hasValidAddressOrNone =
      (postcode.length > 0 && ("" + number).length > 0) || (postcode.length === 0 && ("" + number).length === 0);

    if (this.forms.general.valid && hasValidAddressOrNone) {
      const changes = {};

      if (this.forms.general.get("name").dirty) {
        changes["name"] = name;
      }
      if (this.forms.general.get("postcode").dirty || this.forms.general.get("number").dirty) {
        changes["address"] = {
          postcode: postcode,
          number: parseInt(number, 10),
          numberAddition: numberAddition || null
        };
      }

      this.schoolsService
        .updateInOrganisation(this.organisation.id, { ...changes, id: this.school.id })
        .pipe(take(1))
        .subscribe(
          school => {
            this.toast.success("Aanpassing succesvol opgeslagen");
            this.initialize(school);
          },
          err => {
            this.toast.error("Probleem met opslaan", err.message);
          }
        );
    } else if (!hasValidAddressOrNone) {
      this.toast.error("Adres ongeldig");
    }
  }

  contactClicked(event: any) {
    const contact = event.selected[0] as Contact;

    const modalRef = this.modalService.open(FormModalComponent);
    const component = modalRef.componentInstance as FormModalComponent;
    component.title = "Contact aanpassen";
    component.confirmLabel = "Aanpassen";
    component.fields = [
      new FormModalInputOptions("title", "Titel", "text", contact.title),
      new FormModalInputOptions("firstName", "Voornaam", "text", contact.firstName),
      new FormModalInputOptions("insertion", "Tussenvoegsel", "text", contact.insertion),
      new FormModalInputOptions("lastName", "Achternaam", "text", contact.lastName),
      new FormModalInputOptions("phoneNumber", "Telefoonnummer", "text", contact.phoneNumber),
      new FormModalInputOptions("email", "E-mailadres", "text", contact.email),
      new FormModalInputOptions("receiveEmail", "Ontvangt mails", "switch", contact.receiveEmail),
      new FormModalInputOptions("notes", "Notities", "textarea", contact.notes)
    ];

    component.callback = (result: any) => {
      return this.schoolsService.updateContact(this.organisation.id, this.school.id, {
        ...contact,
        ...result,
        id: contact.id
      });
    };
    component.onSuccess = () => this.initialize(this.schoolsService.currentSchool);
    component.hasDeleteCallback = true;
    component.deleteCallback = () => {
      this.schoolsService.deleteContact(this.organisation.id, this.school.id, contact).subscribe(() => {
        this.toast.success("Contact verwijderd.");
        this.schoolsService
          .getContacts(this.organisation.id, this.school.id)
          .subscribe((contacts: Contact[]) => (this.contacts = contacts));
      });
    };
  }

  newPressed() {
    const modalRef = this.modalService.open(FormModalComponent);
    const component = modalRef.componentInstance as FormModalComponent;
    component.callback = result => this.schoolsService.addContact(this.organisation.id, this.school.id, result);
    component.onSuccess = () => this.initialize(this.schoolsService.currentSchool);
    component.title = "Nieuw contact toevoegen";
    component.confirmLabel = "Toevoegen";
    component.fields = [
      new FormModalInputOptions("title", "Titel", "text", ""),
      new FormModalInputOptions("firstName", "Voornaam", "text", ""),
      new FormModalInputOptions("insertion", "Tussenvoegsel", "text", ""),
      new FormModalInputOptions("lastName", "Achternaam", "text", ""),
      new FormModalInputOptions("phoneNumber", "Telefoonnummer", "text", ""),
      new FormModalInputOptions("email", "E-mailadres", "text", ""),
      new FormModalInputOptions("receiveEmail", "Ontvangt mails", "switch", true),
      new FormModalInputOptions("notes", "Notities", "textarea", "")
    ];
  }
}
