import {Component, OnInit} from '@angular/core';
import {Member} from '@app/shared/models/member.model';
import {NgbModal} from '@ng-bootstrap/ng-bootstrap';
import {SchoolsDataService} from '../../schools-data.service';
import {NotificationsService} from 'angular2-notifications';
import {ActivatedRoute, Router} from '@angular/router';
import {Category} from '@app/shared/models/category.model';
import {CategoriesDataService} from '@app/categories/categories-data.service';
import {FormModalComponent, FormModalInputOptions} from '@app/shared/form-modal/form-modal.component';
import {DropdownOption} from '@app/shared/forms/form-input/form-input.component';
import {FormControl, FormGroup, Validators} from '@angular/forms';
import {DeleteModalComponent} from '@app/shared/delete-modal/modal.component';
import {Group} from '@app/shared/models/group.model';
import {map, take} from 'rxjs/internal/operators';
import {Event} from '@app/shared/models/event.model';
import {Title} from '@angular/platform-browser';
import {EventsDataService} from '@app/events/events-data.service';
import {Organisation} from "@app/shared/models/organisation.model";
import {OrganisationsDataService} from "@app/organisations/organisations-data.service";
import {School} from "@app/shared/models/school.model";
import {of} from "rxjs";

@Component({
  selector: 'app-group-members',
  templateUrl: './group-detail.component.html',
  styleUrls: ['./group-detail.component.scss']
})
export class GroupDetailComponent implements OnInit {
  members: Member[] = [];
  categories: Category[] = [];
  groupId: string;
  group: Group;
  organisation: Organisation = null;
  school: School = null;
  events: Event[];
  eventOptions: DropdownOption[];

  selected: Member[] = [];
  generalInfo: FormGroup;


  constructor(private modal: NgbModal,
              private organisationsService: OrganisationsDataService,
              private schoolsService: SchoolsDataService,
              private eventsService: EventsDataService,
              private toast: NotificationsService,
              private route: ActivatedRoute,
              private router: Router,
              private title: Title,
              private categoriesService: CategoriesDataService) {
  }

  ngOnInit() {
    this.organisation = this.organisationsService.currentOrganisation;
    this.school = this.schoolsService.currentSchool;

    this.route.params.subscribe((params) => {
      if (params.groupId) {
        this.initialize(params.groupId);
      }
    });
  }

  initialize(groupId: string) {
    this.groupId = groupId;
    this.categoriesService.getAllForOrganisation(this.organisation.id).subscribe((categories) => {
      this.categories = categories;
    });

    this.schoolsService.getGroupWithId(this.organisation.id, this.school.id, groupId).subscribe(group => {
      this.group = group;
      this.title.setTitle(`Organisaties > ${this.organisation.name} > Scholen > ${this.school.name} > Groepen > ${group.name}`);

      console.log(group.eventId);

      this.generalInfo = new FormGroup({
        name: new FormControl(this.group.name, Validators.required),
        eventId: new FormControl(this.group.eventId, Validators.required),
      });
    });

    this.eventsService.getAllForOrganisation(this.organisation.id).subscribe(events => {
      this.events = events;
      this.eventOptions = events.map((event) => new DropdownOption(event.id, event.name));
    });

    this.schoolsService.getAllGroupMembers(this.organisation.id, this.school.id, groupId)
      .subscribe((members) => {
        this.members = members;
      });
  }

  newPressed() {
    this.eventsService.getUnassignedMembers(this.organisation.id, this.group.eventId as string, this.school.id).subscribe(unassigned => {
      const modalRef = this.modal.open(FormModalComponent);
      const component = modalRef.componentInstance as FormModalComponent;
      component.title = "Groepslid toevoegen";
      component.confirmLabel = "Toevoegen";

      component.fields = [
        new FormModalInputOptions(
          'userId',
          'Gebruiker',
          'select',
          {
            value: null,
            options: unassigned.map((member) =>
              new DropdownOption(
                member.id,
                member.username +
                (member.person && member.person.firstName ?
                  ` (${member.person.firstName} ${member.person.insertion ?? ""} ${member.person.lastName ?? ""})` : "")
              ))
          },
          [Validators.required])
      ];

      component.callback = (result) => {
        return this.schoolsService.addGroupMember(this.organisation.id, this.school.id, this.groupId, result.userId);
      };
      component.onSuccess = () => {
        this.toast.success("Groepslid succesvol toegevoegd");
        this.initialize(this.groupId);
      };
    });
  }

  updateGeneralInfo() {
    this.schoolsService.updateGroup(this.organisation.id, this.school.id, {
      ...new Group(), ...this.generalInfo.value, id: this.group.id
    }).subscribe(() => {
      this.initialize(this.group.id);
      this.toast.success("Groep succesvol aangepast");
    });
  }

  deletePressed(member: Member) {
    const modalRef = this.modal.open(DeleteModalComponent);
    const component = modalRef.componentInstance as DeleteModalComponent;
    component.name = `Groepslid ${member.username}`;
    modalRef.result.then((result) => {
      this.schoolsService.deleteGroupMember(this.organisation.id, this.school.id, this.groupId, member.id).subscribe(() => {
        this.toast.success(`Groepslid is succesvol verwijderd`);
        this.initialize(this.groupId);
      });
    }).catch(() => {
    });
  }

  editCategoriesPressed() {
    const modalRef = this.modal.open(FormModalComponent);
    const component = modalRef.componentInstance as FormModalComponent;
    component.title = 'Categorieën aanpassen';
    component.confirmLabel = 'Aanpassen';
    component.fields = [
      new FormModalInputOptions(
        'categories',
        'Categorieën',
        'multiselect',
        {
          value: this.group.categories.map(cat => cat.id), options:
            this.categories.map((category) => new DropdownOption(category.id, category.name))
        },
        null,
        {
          size: this.categories.length
        })

    ];
    component.callback = (result) => this.schoolsService.updateGroupCategories(this.organisation.id, this.school.id, this.group.id, result.categories);
    component.onSuccess = () => {
      this.toast.success('Categorieën aangepast');

      this.schoolsService.getGroupWithId(this.organisation.id, this.school.id, this.group.id).subscribe(group => {
        this.group.categories = group.categories;
      });
    };
  }

  rowClicked(event: any) {
    this.router.navigate(["/organisaties", this.organisation.id,'scholen', this.school.id, 'accounts', event.selected[0].id]);
  }
}
