import { Component, OnInit } from "@angular/core";
import { Validators } from "@angular/forms";
import { NgbModal } from "@ng-bootstrap/ng-bootstrap";
import { CrudComponent } from "@app/shared/crud.component";
import { FormModalComponent, FormModalInputOptions } from "@app/shared/form-modal/form-modal.component";
import { Group } from "@app/shared/models/group.model";
import { SchoolsDataService } from "@app/schools/schools-data.service";
import { Event } from "@app/shared/models/event.model";
import { forkJoin, Observable, of } from "rxjs";
import { School } from "@app/shared/models/school.model";
import { DropdownOption } from "@app/shared/forms/form-input/form-input.component";
import { DeleteModalComponent } from "@app/shared/delete-modal/modal.component";
import { NotificationsService } from "angular2-notifications";
import { take, map } from "rxjs/internal/operators";
import { Router } from "@angular/router";
import { Organisation } from "@app/shared/models/organisation.model";
import { OrganisationsDataService } from "@app/organisations/organisations-data.service";
import { EventsDataService } from "@app/events/events-data.service";

@Component({
  selector: "app-groups",
  templateUrl: "./groups.component.html",
  styleUrls: ["./groups.component.scss"],
})
export class GroupsComponent extends CrudComponent<Group> implements OnInit {
  events: Event[];
  organisation: Organisation;
  school: School;
  eventsWithGroups: { event: Event; groups: Observable<Group[]> }[];

  constructor(
    private schoolsService: SchoolsDataService,
    private eventsService: EventsDataService,
    private organisationsService: OrganisationsDataService,
    modal: NgbModal,
    private toast: NotificationsService,
    private router: Router
  ) {
    super();
    super.initialize(modal);
  }

  ngOnInit() {
    this.organisation = this.organisationsService.currentOrganisation;
    this.school = this.schoolsService.currentSchool;
    this.getEvents();
  }

  getEvents() {
    this.eventsService.getAllForOrganisation(this.organisation.id).subscribe((events) => {
      this.events = events;
      this.getEventsWithGroups();
    });
  }

  deletePressed(entity: Group) {
    const modalRef = this.modalService.open(DeleteModalComponent);
    const component = modalRef.componentInstance;
    modalRef.componentInstance.name = entity.name;

    modalRef.result
      .then((result) => {
        this.schoolsService.deleteGroup(this.organisation.id, this.school.id, entity).subscribe(() => {
          this.getEvents();
          this.toast.success("Groep succesvol verwijderd");
        });
      })
      .catch(() => {});
  }

  newPressed() {
    const modalRef = this.modalService.open(FormModalComponent);
    const component = modalRef.componentInstance as FormModalComponent;
    component.title = "Nieuwe groep toevoegen";
    component.description = "Je kunt meerdere groepen tegelijkertijd toevoegen door de namen met komma's te scheiden.";
    component.confirmLabel = "Toevoegen";
    of(this.events)
      .pipe(take(1))
      .subscribe((events) => {
        component.fields = [
          new FormModalInputOptions("name", "Naam van de groep", "text", "", [Validators.required]),
          new FormModalInputOptions(
            "eventId",
            "Evenement",
            "select",
            {
              value: events[0].id,
              options: events.map((event) => new DropdownOption(event.id, event.name)),
            },
            [Validators.required]
          ),
        ];

        component.callback = (result) => {
          const groupsToCreate = result.name
            .split(",")
            .map((name: string) =>
              this.schoolsService.addGroup(this.organisation.id, this.school.id, name.trim(), result.eventId)
            );

          return forkJoin(groupsToCreate);
        };

        component.onSuccess = () => {
          this.getEvents();
          this.toast.success("Groep(en) succesvol aangemaakt");
        };

        component.onError = (err) => {
          this.toast.error("Er ging iets mis.", err.error.message);
        };
      });
  }

  rowClicked(event: any) {
    this.router.navigate([
      "/organisaties",
      this.organisation.id,
      "scholen",
      this.school.id,
      "groepen",
      event.selected[0].id,
    ]);
  }

  getEventsWithGroups() {
    of<Event[]>(this.events)
      .pipe(
        map((events) => {
          setTimeout(() => {
            if (this.table) {
              this.table.messages.totalMessage = "groepen";
            }
          }, 500);
          return events.map((event) => {
            return { event, groups: this.schoolsService.getAllGroups(this.organisation.id, this.school.id, event.id) };
          });
        })
      )
      .subscribe((ewg) => (this.eventsWithGroups = ewg));
  }
}
