import {Component, OnInit} from '@angular/core';
import * as Color from 'color';
import {ActivatedRoute, Router} from '@angular/router';
import {ImporterService} from '@app/schools/importer/importer.service';
import {SchoolsDataService} from '@app/schools/schools-data.service';
import {NotificationsService} from 'angular2-notifications';
import * as XLSX from 'xlsx';
import {Member} from '@app/shared/models/member.model';
import {Group} from '@app/shared/models/group.model';
import {Category} from '@app/shared/models/category.model';
import * as moment from 'moment';
import {Event} from '@app/shared/models/event.model';
import {Title} from '@angular/platform-browser';
import {OrganisationsDataService} from "@app/organisations/organisations-data.service";
import {Organisation} from "@app/shared/models/organisation.model";
import {School} from "@app/shared/models/school.model";
import {EventsDataService} from "@app/events/events-data.service";

class MemberFromImport extends Member {
  password: string;
  categories: Category[];
  groups: Group[];
  extra: string[];
}

@Component({
  selector: 'app-importer',
  templateUrl: './importer.component.html',
  styleUrls: ['./importer.component.scss']
})
export class ImporterComponent implements OnInit {
  color: Color;

  savePersonalData = false;
  payload: any = null;
  result: any = null;

  currentStep = 1;
  steps = [
    'Start',
    'Gegevens markeren',
    'Gegevens corrigeren',
    'Accounts aanmaken',
    'Klaar!'
  ];

  possibleFields = [
    {
      field: '&mdash;',
      description: "Alle kolommen gemarkeerd als \"&mdash;\" worden genegeerd door de server, ze worden niet opgeslagen. Ze worden wel teruggestuurd bij het downloaden van de accountgegevens."
    },
    {
      field: 'Wachtwoord',
      description: "Met deze kolom kunt u per account een wachtwoord instellen. Ontbreekt deze, dan wordt een wachtwoord gegenereerd."
    },
    {
      field: 'Categorie',
      description: "Deze waarde koppelt een categorie aan de leerling. Bestaat de categorie niet, dan krijgt u de optie om een bestaande te selecteren."
    },
    {
      field: 'Groep',
      description: "Deze waarde koppelt een account aan een groep. Bestaat deze groep nog niet, dan krijgt u de keuze om deze aan te maken of om het aan een andere groep toe te voegen."
    }
  ];

  possiblePersonalFields = [
    {field: 'Voornaam, Tussenvoegsel en Achternaam', description: "Naam van de leerling, opgesplitst in drie kolommen."},
    {
      field: 'Geslacht',
      description: "Geslacht van de leerling. Waarde kan Man, Vrouw, of Overig zijn. Afwijkende waarden kunnen later worden gecorrigeerd."
    },
    {field: 'Dieetwensen', description: "Dieetwensen van de leerling. Allergieën en andere bijzonderheden kunnen hier worden opgegeven."},
    {
      field: 'Postcode',
      description: "Postcode van de leerling. Samen met het huisnummer is dit voldoende informatie om het adres automatisch in te vullen. Houdt het formaat \"1234AB\" aan."
    },
    {
      field: 'Huisnummer',
      description: "Huisnummer van de leerling. Samen met de postcode is dit voldoende informatie om het adres automatisch in te vullen. Dit dient alleen een getal te zijn."
    },
    {field: 'Huisnummer toev.', description: "Huisnummer toevoeging."},
    {field: 'Straat', description: "Als u geen postcode en/of huisnummer heeft kunt u hiermee alsnog de straatnaam handmatig invullen."},
    {field: 'Plaats', description: "Als u geen postcode en/of huisnummer heeft kunt u hiermee alsnog de plaatsnaam handmatig invullen."}
  ];

  url: string[];

  importState = 'not_started';

  selectedEvent: Event = null;
  events: Event[] = [];
  organisation: Organisation = null;
  school: School = null;

  constructor(private organisationsService: OrganisationsDataService,
              private schoolsService: SchoolsDataService,
              private eventsService: EventsDataService,
              private helper: ImporterService,
              private route: ActivatedRoute,
              private router: Router,
              private title: Title,
              private toast: NotificationsService) {
  }

  ngOnInit() {
    this.organisation = this.organisationsService.currentOrganisation;
    this.school = this.schoolsService.currentSchool;

    this.helper.initialize();

    this.url = this.router.url.split("?")[0].split("/").slice(1);
    this.url[0] = "/" + this.url[0];

    this.route.queryParams.subscribe((params) => {
      this.currentStep = Number.parseInt(params.stap || 1);
      if (this.helper.data.length === 0) {
        this.router.navigate(this.url, {queryParams: {stap: 1}});
      }
    });

    this.color = this.organisation.primaryColor;

    this.eventsService.getAllForOrganisation(this.organisation.id).subscribe(events => {
      this.events = events;
      if (this.events.length > 0) {
        this.selectedEvent = this.events[0];
      }
    });

    this.title.setTitle(`Scholen > ${this.school.name} > Importeren`);
  }

  onFileChange(evt: any) {
    this.helper.changeFile(evt);
  }

  isFileSet() {
    return this.helper.loadedDataAsJSON !== '';
  }

  onSubmitFile() {
    this.helper.submitFile(this.savePersonalData, this.selectedEvent);
    this.router.navigate(this.url, {queryParams: {stap: 2}});
  }

  onSubmitOptions() {
    this.helper.submitOptions().subscribe(() => {
      if (this.isEverytingCorrected()) {
        this.helper.submitCorrections().subscribe((payload) => {
          this.payload = payload;
          this.router.navigate(this.url, {queryParams: {stap: 4}});
        });
      } else {
        this.router.navigate(this.url, {queryParams: {stap: 3}});
      }
    });
  }

  onSubmitCorrections() {
    this.helper.submitCorrections().subscribe(payload => {
      this.payload = payload;
      this.router.navigate(this.url, {queryParams: {stap: 4}});
    });
  }

  onSubmitImport() {
    this.importState = 'started';
    this.schoolsService.importStudents(this.organisation.id, this.school.id, {...this.payload, eventId: this.selectedEvent.id}).subscribe((result) => {
      this.result = result;
      this.importState = "not_started";
      this.toast.success('Importeren geslaagd.');
      this.router.navigate(this.url, {queryParams: {stap: 5}});
    }, (error) => {
      console.log(error);
      this.toast.error('Importeren niet geslaagd.');
    });
  }

  onDownloadImportFile() {
    const filename = 'JBD-Import-' + (moment(new Date()).format('YYYYMMDD')) + '-Accountgegevens.xlsx';
    const hasPersonalData = this.result.personalData;

    const headers = [
      'Gebruikersnaam',
      'Wachtwoord',
      'Categorie',
      'Groep',
      ...(hasPersonalData ? [
        'Voornaam',
        'Tussenvoegsel',
        'Achternaam',
        'Geslacht',
        'Dieetwensen',
        'Straat',
        'Huisnummer',
        'Huisnummer toev.',
        'Postcode',
        'Plaats'] : []),
      'ID',
      ...(this.result.members[0].extra ? Object.keys(this.result.members[0].extra) : [])
    ];
    const data = [headers];
    this.result.members.forEach((member: MemberFromImport) => {
      const row: string[] = [];
      row.push(member.username);
      row.push(member.password);
      row.push(member.categories.length > 0 ? member.categories[0].name : '');
      row.push(member.groups.length > 0 ? member.groups[0].name : '');
      if (hasPersonalData) {
        row.push(member.person.firstName || '');
        row.push(member.person.insertion || '');
        row.push(member.person.lastName || '');
        row.push(member.person.gender ? (member.person.gender === 'male' ? 'Man' : 'Vrouw') : 'Onbekend');
        row.push(member.person.dietInfo || '');
        if (member.person.address) {
          row.push(member.person.address.street || '');
          row.push(('' + member.person.address.number) || '');
          row.push(member.person.address.numberAddition || '');
          row.push(member.person.address.postcode || '');
          row.push(member.person.address.city || '');
        } else {
          for (let i = 0; i < 5; ++i) {
            row.push("");
          }
        }
      }
      row.push(member.id);
      (member.extra ? Object.values(member.extra) : []).forEach(extra => row.push(extra));
      data.push(row);
    });
    const ws_name = 'Accountgegevens';
    const wb = XLSX.utils.book_new(), ws = XLSX.utils.aoa_to_sheet(data);

    XLSX.utils.book_append_sheet(wb, ws, ws_name);
    XLSX.writeFile(wb, filename);
  }

  onModifierChange(evt: any, columnIndex: number) {
    this.helper.onModifierChange(evt, columnIndex);
  }

  isEverytingCorrected() {
    const invalids = (this.helper.getMismatchedGroups().filter(mismatch => !mismatch.correction).length +
      this.helper.getMismatchedCategories().filter(mismatch => !mismatch.correction).length +
      this.helper.getMismatchedGenders().filter(mismatch => !mismatch.correction).length);
    return invalids === 0;
  }

  getMismatchedGroups() {
    return this.helper.getMismatchedGroups();
  }

  getMismatchedCategories() {
    return this.helper.getMismatchedCategories();
  }

  getMismatchedGenders() {
    return this.helper.getMismatchedGenders();
  }

  getGroups() {
    return this.helper.groups$;
  }

  getCategories() {
    return this.helper.categories$;
  }

  getGenders() {
    return this.helper.genders;
  }

  getHeaders(): string[] {
    return this.helper.headers;
  }

  getModifiers(index: number) {
    return this.helper.getModifiers(index);
  }

  makeCorrection(type: string, original: string, event: any) {
    this.helper.makeCorrection(type, original, event);
  }

  getColumnModifierValue(index: number) {
    return this.helper.getColumnModifierValue(index);
  }

  getWarnings() {
    const {members, warnings} = this.result;

    return this.result.warnings.map((warningObject: any) => {
      const user = members[warningObject.index];
      let warning: string = warningObject.warning;
      switch (warning) {
        case "addressInvalid":
          warning = "Adres is ongeldig";
          break;
        case "invalidCategories":
          warning = "Categorie is ongeldig";
          break;
        case "invalidGroups":
          warning = "Groep is ongeldig";
          break;
        default:
          break;
      }

      return {user, warning};
    });
  }

  goToStep(step: number) {
    this.router.navigate(this.url, {queryParams: {stap: step}});
  }
}
