import { Injectable } from "@angular/core";
import { forkJoin, merge, Observable, of } from "rxjs";
import { Group } from "@app/shared/models/group.model";
import { Category } from "@app/shared/models/category.model";
import * as XLSX from "xlsx";
import { filter, flatMap, switchMap, take, tap } from "rxjs/internal/operators";
import { Event } from "@app/shared/models/event.model";
import { SchoolsDataService } from "@app/schools/schools-data.service";
import { CategoriesDataService } from "@app/categories/categories-data.service";
import { OrganisationsDataService } from "@app/organisations/organisations-data.service";
import { Organisation } from "@app/shared/models/organisation.model";
import { School } from "@app/shared/models/school.model";

class ColumnModifier {
  required = false;
  columnIndex = -1;

  constructor(public value: string, public label: string, required: boolean = false, currentColumn: number = -1) {}
}

@Injectable()
export class ImporterService {
  loadedDataAsJSON = "";
  data: any = [];
  headers: string[];
  columnCount = 0;

  payload: any = null;

  columnModifiers: Array<ColumnModifier> = [];
  personalColumnModifiers: Array<ColumnModifier> = [];
  columnModifierValues: Array<string> = [];

  genders: { id: string; name: string }[] = [
    { id: "male", name: "Man" },
    { id: "female", name: "Vrouw" },
    { id: "other", name: "Overig" },
  ];

  categories$: Observable<Category[]>;
  groups$: Observable<Group[]>;

  mismatchedCategories: { original: string; correction: { id: string; name: string } }[] = [];
  mismatchedGroups: { original: string; correction: { id: string; name: string } }[] = [];
  mismatchedGenders: { original: string; correction: { id: string; name: string } }[] = [];

  savePersonalData = false;

  selectedEvent: Event = null;
  organisation: Organisation = null;
  school: School = null;

  constructor(
    private organisationsService: OrganisationsDataService,
    private schoolsService: SchoolsDataService,
    private categoriesService: CategoriesDataService,
    private schoolService: SchoolsDataService
  ) {}

  initialize() {
    this.organisation = this.organisationsService.currentOrganisation;
    this.categories$ = this.categoriesService.getAllForOrganisation(this.organisation.id);

    this.school = this.schoolsService.currentSchool;
  }

  initColumns() {
    const oldColumnCount = this.columnCount;
    this.data.forEach((row: Array<any>) => (this.columnCount = Math.max(row.length, this.columnCount)));
    this.data.map((row: any[]) => {
      while (this.columnCount > row.length) {
        row.push("");
      }
    });

    this.columnModifierValues = [];
    for (let i = 0; i < this.columnCount; ++i) {
      this.columnModifierValues.push("");
    }

    this.headers = this.data[0];
    this.headers.forEach((col: string, index: number) => {
      const availableModifiers = this.getModifiers(index);
      const modifier = availableModifiers.find((opt) => opt.label === col || opt.value === col);
      if (modifier) {
        modifier.columnIndex = index;
        this.columnModifierValues[index] = modifier.value;
      }
    });
  }

  initColumnModifiers(savePersonalData: boolean) {
    this.personalColumnModifiers = [
      new ColumnModifier("person.firstName", "Voornaam", false),
      new ColumnModifier("person.insertion", "Tussenvoegsel", false),
      new ColumnModifier("person.lastName", "Achternaam", false),
      new ColumnModifier("person.dateOfBirth", "Geboortedatum", false),
      new ColumnModifier("person.gender", "Geslacht", false),
      new ColumnModifier("person.address.postcode", "Postcode", false),
      new ColumnModifier("person.address.number", "Huisnummer", false),
      new ColumnModifier("person.address.numberAddition", "Huisnummer toev.", false),
      new ColumnModifier("person.dietInfo", "Dieetwensen", false),
    ];

    this.columnModifiers = [
      new ColumnModifier("category", "Categorie", false),
      new ColumnModifier("group", "Groep", false),
      new ColumnModifier("password", "Wachtwoord", false),
      ...(savePersonalData ? this.personalColumnModifiers : []),
    ];
  }

  changeFile(evt: any) {
    this.data = null;
    this.columnCount = 0;

    /* wire up file reader */
    const target: DataTransfer = <DataTransfer>evt.target;
    if (target.files.length !== 1) {
      throw new Error("Cannot use multiple files");
    }
    const reader: FileReader = new FileReader();
    reader.onload = (e: any) => {
      /* read workbook */
      const bstr: string = e.target.result;
      const wb: XLSX.WorkBook = XLSX.read(bstr, { type: "binary" });

      /* grab first sheet */
      const wsname: string = wb.SheetNames[0];
      const ws: XLSX.WorkSheet = wb.Sheets[wsname];

      /* save data */
      this.data = XLSX.utils.sheet_to_json(ws, { header: 1 });
      this.loadedDataAsJSON = JSON.stringify(this.data);
    };
    reader.readAsBinaryString(target.files[0]);
  }

  onModifierChange(evt: any, columnIndex: number) {
    const newColumnModifierValue = evt.target.value;

    const oldValue = this.columnModifierValues[columnIndex];
    this.columnModifiers.forEach((colOpt) => {
      if (colOpt.value === oldValue) {
        colOpt.columnIndex = -1;
      }
      if (colOpt.value === newColumnModifierValue) {
        colOpt.columnIndex = columnIndex;
      }
    });
    this.columnModifierValues[columnIndex] = newColumnModifierValue;
  }

  getModifiers(colIndex: number) {
    return this.columnModifiers.filter((option) => option.columnIndex === -1 || option.columnIndex === colIndex);
  }

  submitFile(savePersonalData: boolean, event: Event) {
    this.savePersonalData = savePersonalData;
    this.selectedEvent = event;
    this.groups$ = this.schoolService.getAllGroups(this.organisation.id, this.school.id, event.id);
    this.initColumnModifiers(savePersonalData);
    this.data = JSON.parse(this.loadedDataAsJSON);
    this.initColumns();
    this.data.splice(0, 1);
  }

  submitOptions(): Observable<boolean[]> {
    const mismatches: Observable<boolean>[] = [];

    // Cleanup mismatches
    this.mismatchedGroups = [];
    this.mismatchedCategories = [];
    this.mismatchedGenders = [];

    const catColumnIndex = this.columnModifiers.find((option) => option.value === "category").columnIndex;
    if (catColumnIndex !== -1) {
      mismatches.push(
        this.categories$.pipe(
          take(1),
          switchMap((cats: Category[]) => {
            if (cats.length === 0) {
              return of(true);
            }
            this.data.forEach((row: any[]) => {
              const catEntity = cats.find((cat) => cat.name === row[catColumnIndex]);
              if (!catEntity) {
                this.mismatchedCategories[row[catColumnIndex]] = {
                  original: row[catColumnIndex] as string,
                  correction: null,
                };
              } else {
                row[catColumnIndex] = catEntity;
              }
            });
            return of(true);
          })
        )
      );
    }

    const groupColumnIndex = this.columnModifiers.find((option) => option.value === "group").columnIndex;
    if (groupColumnIndex !== -1) {
      mismatches.push(
        this.groups$.pipe(
          take(1),
          switchMap((groups: Group[]) => {
            this.data.forEach((row: any[]) => {
              const groupEntity = groups.find((group) => group.name === row[groupColumnIndex]);
              if (!groupEntity) {
                this.mismatchedGroups[row[groupColumnIndex]] = {
                  original: row[groupColumnIndex] as string,
                  correction: null,
                };
              } else {
                row[groupColumnIndex] = groupEntity;
              }
            });
            return of(true);
          })
        )
      );
    }

    if (this.savePersonalData) {
      const genderColumnIndex = this.columnModifiers.find((option) => option.value === "person.gender").columnIndex;
      if (genderColumnIndex !== -1) {
        this.data.forEach((row: any[]) => {
          const genderEntity = this.genders.find((gender) => gender.name === row[genderColumnIndex]);
          if (!genderEntity) {
            this.mismatchedGenders[row[genderColumnIndex]] = {
              original: row[genderColumnIndex] as string,
              correction: null,
            };
          } else {
            row[genderColumnIndex] = genderEntity;
          }
        });
      }
    }

    return forkJoin(mismatches);
  }

  submitCorrections() {
    const groupColumn = this.columnModifiers.find((option) => option.value === "group");
    const catColumn = this.columnModifiers.find((option) => option.value === "category");
    const genderColumn = this.columnModifiers.find((option) => option.value === "person.gender");

    const groupColumnIndex = groupColumn ? groupColumn.columnIndex : undefined;
    const catColumnIndex = catColumn ? catColumn.columnIndex : undefined;
    const genderColumnIndex = genderColumn ? genderColumn.columnIndex : undefined;

    const mismatchedGenders = this.getMismatchedGenders();
    const mismatchedGroups = this.getMismatchedGroups();
    const mismatchedCategories = this.getMismatchedCategories();

    this.data = this.data.map((row: any[]) => {
      if (groupColumnIndex !== undefined) {
        const groupCorrection = mismatchedGroups.find((group) => group.original === row[groupColumnIndex]);
        if (groupCorrection) {
          row[groupColumnIndex] = groupCorrection.correction;
        }
      }

      if (catColumnIndex !== undefined) {
        const categoryCorrection = mismatchedCategories.find((cat) => cat.original === row[catColumnIndex]);
        if (categoryCorrection) {
          row[catColumnIndex] = categoryCorrection.correction;
        }
      }

      if (genderColumnIndex !== undefined) {
        const genderCorrection = mismatchedGenders.find((gender) => gender.original === row[genderColumnIndex]);
        if (genderCorrection) {
          row[genderColumnIndex] = genderCorrection.correction;
        }
      }

      return row;
    });

    return this.mapToRequestPayload().pipe(
      tap((payload) => {
        this.payload = payload;
      })
    );
  }

  createNewGroupsIfNecessary(): Observable<any> {
    const groupsToCreate = this.getMismatchedGroups().filter((g) => g.correction && g.correction.id === "newGroup");
    if (groupsToCreate.length > 0) {
      let counter = 0;
      return of(this.selectedEvent).pipe(
        switchMap((event) =>
          merge(
            groupsToCreate.map((g) => {
              return this.schoolService.addGroup(this.organisation.id, this.school.id, g.original, event.id);
            })
          )
        ),
        flatMap((group) => group),
        take(groupsToCreate.length),
        switchMap(() => of(++counter === groupsToCreate.length))
      );
    } else {
      return of(true);
    }
  }

  mapToRequestPayload(): Observable<any> {
    const indexMap = this.getModifierIndexMap();

    return this.createNewGroupsIfNecessary().pipe(
      filter((isDone: boolean) => isDone),
      flatMap(() => this.schoolService.getAllGroups(this.organisation.id, this.school.id, this.selectedEvent.id)),
      switchMap((allGroups) => {
        console.log("allGroups", allGroups);
        return of({
          personalData: this.savePersonalData,
          data: this.data.map((row: any[]) => {
            const extra = this.getExtraDataForRow(row);
            const categories = indexMap["category"] !== -1 ? [Number.parseInt(row[indexMap["category"]].id)] : [];
            const groupNameToMatch = row[indexMap["group"]].name;
            const groups =
              indexMap["group"] !== -1 ? [(allGroups || []).find((g) => groupNameToMatch === g.name)?.id] : [];
            const password = indexMap["password"] !== -1 ? row[indexMap["password"]] : "";
            const person = !this.savePersonalData
              ? null
              : {
                  firstName: indexMap["person.firstName"] !== -1 ? row[indexMap["person.firstName"]] : null,
                  insertion: indexMap["person.insertion"] !== -1 ? row[indexMap["person.insertion"]] : null,
                  lastName: indexMap["person.lastName"] !== -1 ? row[indexMap["person.lastName"]] : null,
                  dateOfBirth: indexMap["person.dateOfBirth"] !== -1 ? row[indexMap["person.dateOfBirth"]] : null,
                  gender: indexMap["person.gender"] !== -1 ? row[indexMap["person.gender"]].id : null,
                  dietInfo: indexMap["person.dietInfo"] !== -1 ? row[indexMap["person.dietInfo"]] : null,
                  address: {
                    postcode:
                      indexMap["person.address.postcode"] !== -1 ? row[indexMap["person.address.postcode"]] : null,
                    number: indexMap["person.address.number"] !== -1 ? row[indexMap["person.address.number"]] : null,
                    numberAddition:
                      indexMap["person.address.numberAddition"] !== -1
                        ? row[indexMap["person.address.numberAddition"]]
                        : null,
                  },
                };
            return {
              person,
              extra,
              categories,
              groups,
              password,
            };
          }),
        });
      })
    );
  }

  makeCorrection(type: string, original: string, event: any) {
    let record = null;

    switch (type) {
      case "groups":
        record = this.getMismatchedGroups().find((mismatch) => mismatch.original === original);
        break;
      case "categories":
        record = this.getMismatchedCategories().find((mismatch) => mismatch.original === original);
        break;
      case "genders":
        record = this.getMismatchedGenders().find((mismatch) => mismatch.original === original);
        break;
    }

    const idAndName = event.target.value.split("|");
    record.correction = { id: idAndName[0], name: idAndName[1] };
  }

  getMismatchedGroups() {
    return Object.entries(this.mismatchedGroups).map((entry) => entry[1]);
  }

  getMismatchedCategories() {
    return Object.entries(this.mismatchedCategories).map((entry) => entry[1]);
  }

  getMismatchedGenders() {
    return Object.entries(this.mismatchedGenders).map((entry) => entry[1]);
  }

  getModifierIndexMap() {
    const options = {};
    this.columnModifiers.forEach((option) => {
      options[option.value] = option.columnIndex;
    });
    return options;
  }

  getHeadersWithOptions() {
    return this.headers.map((col, idx) => {
      return { header: col, option: this.columnModifiers.find((c) => c.columnIndex === idx) };
    });
  }

  getExtraDataForRow(row: any[]) {
    const headersWithOptions = this.getHeadersWithOptions();

    const extraValues = row
      .map((value, index) => ({ value, index }))
      .filter((col) => !headersWithOptions[col.index].option);

    const extras = {};
    extraValues.forEach((extra) => (extras[this.headers[extra.index]] = extra.value));

    return extras;
  }

  getColumnModifierValue(index: number) {
    return this.columnModifierValues[index];
  }
}
